package com.tbyf.cdcengine2.core;

import io.debezium.embedded.Connect;
import io.debezium.engine.DebeziumEngine;
import io.debezium.engine.spi.OffsetCommitPolicy;
import io.debezium.relational.history.FileDatabaseHistory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.atomic.AtomicLong;

public abstract class AbstractCdcEngine<T extends AbstractCdcEngine<T>> {

    private final static AtomicLong nextConsumerId = new AtomicLong(0);

    private final Logger logger = LoggerFactory.getLogger(getClass());

    private final Object startupStopMonitor = new Object();

    private volatile DebeziumEngine<?> debeziumEngine;

    private volatile ExecutorService executor;

    protected final Properties debeziumProps = new Properties();

    private final BlockingQueue<ChangedRecord> recordQueue = new ArrayBlockingQueue<>(10000);

    protected ChangeEventAdapter adapter() {
        return DefaultChangeEventAdapter.INSTANCE;
    }

    private volatile Thread consumerThread;

    private volatile boolean running = true;

    private ChangeHandler changeHandler;

    private Runnable errorHandler;

    public AbstractCdcEngine() {
        debeziumProps.setProperty(Constants.DATABASE_SERVER_NAME_PROP, Constants.DATABASE_SERVER_NAME_VALUE);
        debeziumProps.setProperty(Constants.DATABASE_HISTORY_PROP, FileDatabaseHistory.class.getName());
        debeziumProps.setProperty("name", getClass().getSimpleName());
    }

    @SuppressWarnings("unchecked")
    private T self() {
        return (T) this;
    }

    public T onChange(ChangeHandler changeHandler) {
        if (changeHandler == null) {
            throw new IllegalArgumentException("changeHandler must not be null");
        }
        this.changeHandler = changeHandler;
        return self();
    }

    public T onError(Runnable errorHandler) {
        this.errorHandler = errorHandler;
        return self();
    }

    public void start() {
        synchronized (startupStopMonitor) {
            logger.info("开始启动CDC引擎...");
            if (this.changeHandler == null) {
                throw new IllegalStateException("changeHandler not set");
            }
            this.running = true;
            consumerThread = new Thread(() -> {
                while (running) {
                    try {
                        List<ChangedRecord> buffer = new ArrayList<>();
                        recordQueue.drainTo(buffer);
                        if (buffer.isEmpty()) {
                            ChangedRecord record = recordQueue.take();
                            try {
                                this.changeHandler.handle(record);
                            } catch (Exception e) {
                                logger.error("处理{}时发生了异常", record, e);
                            }
                        } else {
                            for (ChangedRecord record : buffer) {
                                try {
                                    changeHandler.handle(record);
                                } catch (Exception e) {
                                    logger.error("处理{}时发生了异常", record, e);
                                }
                            }
                        }
                    } catch (InterruptedException ignored) {
                        break;
                    }
                }
                // 消费完剩余的记录
                List<ChangedRecord> buffer = new ArrayList<>();
                recordQueue.drainTo(buffer);
                if (!buffer.isEmpty()) {
                    for (ChangedRecord record : buffer) {
                        try {
                            changeHandler.handle(record);
                        } catch (Exception e) {
                            logger.error("处理{}时发生了异常", record, e);
                        }
                    }
                }

            }, "cdc-consumer-" + nextConsumerId.getAndIncrement());
            consumerThread.start();

            debeziumEngine = DebeziumEngine.create(Connect.class)
                    .using(OffsetCommitPolicy.always())
                    .using(debeziumProps)
                    .using(((success, message, error) -> {
                        if (!success) {
                            logger.error("启动失败, 原因: {}", message, error);
                            stop();
                        }
                    }))
                    .notifying(event -> {
                        try {
                            ChangedRecord changedRecord = adapter().adapt(event);
                            if (changedRecord != null) {
                                recordQueue.put(changedRecord);
                            }
                        } catch (InterruptedException ignored) {
                        } catch (Exception e) {
                            logger.error(e.getMessage(), e);
                        }
                    }).build();

            executor = Executors.newSingleThreadExecutor();
            executor.execute(debeziumEngine);
        }
    }

    public void stop() {
        synchronized (startupStopMonitor) {
            logger.info("开始关闭CDC引擎...");
            running = false;
            DebeziumEngine<?> debeziumEngine = this.debeziumEngine;
            if (debeziumEngine != null) {
                try {
                    debeziumEngine.close();
                } catch (IOException e) {
                    logger.error("关闭时发生了错误", e);
                }
            }
            this.debeziumEngine = null;

            ExecutorService executor = this.executor;
            if (executor != null) {
                executor.shutdownNow();
            }
            this.executor = null;

            Thread consumerThread = this.consumerThread;
            if (consumerThread != null) {
                consumerThread.interrupt();
            }
            this.consumerThread = null;

            Runnable errorHandler = this.errorHandler;
            if (errorHandler != null) {
                errorHandler.run();
            }
        }
    }

}
