package com.tbyf.cdcengine2.core;

import io.debezium.data.Envelope;
import io.debezium.data.VariableScaleDecimal;
import io.debezium.engine.ChangeEvent;
import io.debezium.time.Timestamp;
import org.apache.kafka.connect.data.Field;
import org.apache.kafka.connect.data.Schema;
import org.apache.kafka.connect.data.Struct;
import org.apache.kafka.connect.source.SourceRecord;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.util.HashMap;
import java.util.Map;

public class DefaultChangeEventAdapter implements ChangeEventAdapter {

    public static final DefaultChangeEventAdapter INSTANCE = new DefaultChangeEventAdapter();

    @Override
    public ChangedRecord adapt(ChangeEvent<SourceRecord, SourceRecord> event) throws Exception {
        SourceRecord sr = event.value();
        Struct value = (Struct) sr.value();
        if (value == null) {
            return null;
        }
        Schema valueSchema = sr.valueSchema();
        if (!valueSchema.name().endsWith(".Envelope")) {
            return null;
        }

        ChangedRecord record = new ChangedRecord();
        String destination = event.destination();
        String schemaTable = destination.substring(destination.indexOf(".") + 1);
        String[] split = schemaTable.split("\\.");
        record.schema(split[0]);
        record.table(split[1]);
        Struct before = value.getStruct(valueSchema.field(Envelope.FieldName.BEFORE).name());
        Struct now = value.getStruct(valueSchema.field(Envelope.FieldName.AFTER).name());
        String op = value.getString(valueSchema.field(Envelope.FieldName.OPERATION).name());
        Envelope.Operation operation = Envelope.Operation.forCode(op);
        if (operation == Envelope.Operation.CREATE) {
            record.operationType(OperationType.CREATE);
        } else if (operation == Envelope.Operation.UPDATE) {
            record.operationType(OperationType.UPDATE);
        } else if (operation == Envelope.Operation.DELETE) {
            record.operationType(OperationType.DELETE);
        } else {
            return null;
        }
        record.before(resolveColumns(before))
                .now(resolveColumns(now));
        return record;
    }

    private Map<String, Object> resolveColumns(Struct cols) {
        if (cols == null) {
            return null;
        }
        HashMap<String, Object> result = new HashMap<>();
        Schema schema = cols.schema();
        for (Field field : schema.fields()) {
            Object value = cols.get(field);
            if (value != null) {
                if (field.schema().type() == Schema.Type.STRUCT) {
                    Struct structValue = (Struct) value;
                    // 处理小数类型
                    if (field.schema().name().equals(VariableScaleDecimal.class.getName())) {
                        value = new BigDecimal(new BigInteger(structValue.getBytes(VariableScaleDecimal.VALUE_FIELD)),
                                structValue.getInt32(VariableScaleDecimal.SCALE_FIELD));
                    }
                }
                if (field.schema().type() == Schema.Type.INT64) {
                    Long longValue = (Long) value;
                    // 处理时间类型
                    if (field.schema().name().equals(Timestamp.class.getName())) {
                        value = LocalDateTime.ofInstant(Instant.ofEpochMilli(longValue), ZoneOffset.UTC);
                    }
                }
                // todo 考虑其他情况
            }
            result.put(field.name(), value);
        }
        return result;
    }
}
