package com.tbyf.cdcengine2.oracle;

import com.tbyf.cdcengine2.core.AbstractCdcEngine;
import com.tbyf.cdcengine2.core.ChangeEventAdapter;
import com.tbyf.cdcengine2.core.DefaultChangeEventAdapter;
import io.debezium.connector.oracle.OracleConnector;
import io.debezium.relational.history.FileDatabaseHistory;

import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.SQLException;
import java.util.Arrays;
import java.util.HashSet;
import java.util.stream.Collectors;

public class OracleCdcEngine extends AbstractCdcEngine<OracleCdcEngine> {

    static {
        try {
            Class.forName("oracle.jdbc.driver.OracleDriver");
        } catch (ClassNotFoundException e) {
            throw new IllegalStateException("未找到Oracle jdbc驱动", e);
        }
    }

    public OracleCdcEngine(OracleCdcEngineProperties props) {
        debeziumProps.setProperty("connector.class", OracleConnector.class.getName());
        debeziumProps.setProperty("database.history", FileDatabaseHistory.class.getName());
        debeziumProps.setProperty("database.history.file.filename", props.namespace() + "_his.dat");
        debeziumProps.setProperty("offset.storage.file.filename", props.namespace() + "_off.dat");

        debeziumProps.setProperty("database.hostname", props.host());
        debeziumProps.setProperty("database.port", String.valueOf(props.port()));
        debeziumProps.setProperty("database.user", props.username());
        debeziumProps.setProperty("database.password", props.password());
        debeziumProps.setProperty("database.dbname", props.dbname());

        debeziumProps.setProperty("log.mining.strategy", "online_catalog");
        debeziumProps.setProperty("snapshot.mode", "schema_only");

        debeziumProps.setProperty("database.history.skip.unparseable.ddl", Boolean.toString(true));
        debeziumProps.setProperty("database.history.store.only.captured.tables.ddl", Boolean.toString(true));

        debeziumProps.setProperty("database.server.name", "server1");
        debeziumProps.setProperty("name", "oracle_cdc_engine");

        String schema = props.username().toUpperCase();
        debeziumProps.setProperty("schema.include.list", schema);

        String tables = new HashSet<>(Arrays.asList(props.cdcTables().toUpperCase().split(",")))
                .stream().map(tableName -> schema + "." + tableName)
                .collect(Collectors.joining(","));
        debeziumProps.setProperty("table.include.list", tables);


        // 开启supplemental logging
        String url = buildOracleJdbcUrl(props.host(), props.port(), props.dbname());
        SupplementalLoggingUtils.enableDBSupplementalLogging(getConn(url, props.username(), props.password()));
        SupplementalLoggingUtils.enableSchemaTablesSupplementalLogging(getConn(url, props.username(), props.password()), props.username(),
                new HashSet<>(Arrays.asList(props.cdcTables().split(","))));
    }

    private static Connection getConn(String url, String username, String password) {
        try {
            return DriverManager.getConnection(url, username, password);
        } catch (SQLException e) {
            throw new IllegalStateException("获取数据库连接失败", e);
        }
    }

    public static String buildOracleJdbcUrl(String host, int port, String dbname) {
        return String.format("jdbc:oracle:thin:@%s:%d:%s", host, port, dbname);
    }

}
