package com.tbyf.cdcengine2.core;

import io.debezium.embedded.Connect;
import io.debezium.engine.DebeziumEngine;
import io.debezium.engine.spi.OffsetCommitPolicy;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

public abstract class AbstractCdcEngine<T extends AbstractCdcEngine<T>> {

    private final Logger logger = LoggerFactory.getLogger(getClass());

    private final Object startupStopMonitor = new Object();

    private volatile DebeziumEngine<?> debeziumEngine;

    private volatile ExecutorService executor;

    protected final Properties debeziumProps = new Properties();

    private final BlockingQueue<ChangedRecord> recordQueue = new ArrayBlockingQueue<>(10000);

    protected ChangeEventAdapter adapter() {
        return DefaultChangeEventAdapter.INSTANCE;
    }

    private volatile Thread consumerThread;

    private ChangeHandler handler;

    private volatile boolean running = true;

    public interface ChangeHandler {
        void process(ChangedRecord record) throws Exception;
    }

    @SuppressWarnings("unchecked")
    public T changeHandler(ChangeHandler handler) {
        this.handler = handler;
        return (T) this;
    }

    public void start() {
        synchronized (startupStopMonitor) {
            ChangeHandler handler0 = this.handler;
            if (handler0 == null) {
                throw new IllegalStateException("recordConsumer not set");
            }
            consumerThread = new Thread(() -> {
                while (running) {
                    try {
                        List<ChangedRecord> buffer = new ArrayList<>();
                        recordQueue.drainTo(buffer);
                        if (buffer.isEmpty()) {
                            ChangedRecord record = recordQueue.take();
                            try {
                                handler0.process(record);
                            } catch (Exception e) {
                                logger.error("处理{}时发生了异常", record, e);
                            }
                        } else {
                            for (ChangedRecord record : buffer) {
                                try {
                                    handler0.process(record);
                                } catch (Exception e) {
                                    logger.error("处理{}时发生了异常", record, e);
                                }
                            }
                        }
                    } catch (InterruptedException e) {
                    }
                }
                // 消费完剩余的记录
                List<ChangedRecord> buffer = new ArrayList<>();
                recordQueue.drainTo(buffer);
                if (!buffer.isEmpty()) {
                    for (ChangedRecord record : buffer) {
                        try {
                            handler0.process(record);
                        } catch (Exception e) {
                            logger.error("处理{}时发生了异常", record, e);
                        }
                    }
                }

            }, "cdc-consumer");
            consumerThread.start();

            debeziumEngine = DebeziumEngine.create(Connect.class)
                    .using(OffsetCommitPolicy.always())
                    .using(debeziumProps)
                    .using(((success, message, error) -> {
                        if (!success) {
                            logger.error("启动失败, 原因: {}", message, error);
                            stop();
                        }
                    }))
                    .notifying(event -> {
                        try {
                            ChangedRecord changedRecord = adapter().adapt(event);
                            if (changedRecord != null) {
                                recordQueue.put(changedRecord);
                            }
                        } catch (InterruptedException ignored) {
                        } catch (Exception e) {
                            logger.error(e.getMessage(), e);
                        }
                    }).build();

            executor = Executors.newSingleThreadExecutor();
            executor.execute(debeziumEngine);
        }
    }

    public void stop() {
        synchronized (startupStopMonitor) {
            running = false;
            DebeziumEngine<?> debeziumEngine = this.debeziumEngine;
            if (debeziumEngine != null) {
                try {
                    debeziumEngine.close();
                } catch (IOException e) {
                    logger.error("关闭时发生了错误", e);
                }
            }
            this.debeziumEngine = null;

            ExecutorService executor = this.executor;
            if (executor != null) {
                executor.shutdown();
            }
            this.executor = null;

            Thread consumerThread = this.consumerThread;
            if (consumerThread != null && consumerThread.isAlive()) {
                consumerThread.interrupt();
            }
            this.consumerThread = null;
        }
    }
}
