package com.tbyf.cdcengine2.oracle;

import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 开启数据库或表的supplemental logging的工具类
 */
public class SupplementalLoggingUtils {

    /**
     * 查询数据库supplemental logging开启状态SQL
     */
    private static final String QUERY_DB_SUPPLEMENTAL_LOGGING_SQL = "SELECT SUPPLEMENTAL_LOG_DATA_MIN FROM V$DATABASE";

    /**
     * 开启数据库supplemental logging SQL
     */
    private static final String ENABLE_DB_SUPPLEMENTAL_LOGGING_SQL = "ALTER DATABASE ADD SUPPLEMENTAL LOG DATA";

    /**
     * 查询所有已开启了supplemental logging的表的SQL
     */
    private static final String QUERY_TABLE_SUPPLEMENTAL_LOGGING_SQL = "SELECT OWNER, OWNER||'.'||TABLE_NAME AS TABLE_NAME FROM ALL_LOG_GROUPS WHERE LOG_GROUP_TYPE = 'ALL COLUMN LOGGING'";

    /**
     * 开启表的supplemental logging SQL
     */
    private static final String ENABLE_TABLE_SUPPLEMENTAL_LOGGING_SQL = "ALTER TABLE %s ADD SUPPLEMENTAL LOG DATA (ALL) COLUMNS";

    /**
     * 开启Oracle数据库supplemental logging
     *
     * @param conn
     */
    public static void enableDBSupplementalLogging(Connection conn) {
        try {
            if (!isDBSupplementalLoggingEnabled(conn)) {
                try (Statement stmt = conn.createStatement()) {
                    stmt.executeUpdate(ENABLE_DB_SUPPLEMENTAL_LOGGING_SQL);
                }
            }
        } catch (Exception e) {
            throw new IllegalStateException("开启数据库supplemental logging时发生了异常", e);
        }
    }

    private static boolean isDBSupplementalLoggingEnabled(Connection conn) throws SQLException {
        try (Statement stmt = conn.createStatement()) {
            try (ResultSet rs = stmt.executeQuery(QUERY_DB_SUPPLEMENTAL_LOGGING_SQL)) {
                if (rs.next()) {
                    String result = rs.getString(1);
                    return "YES".equals(result) || "IMPLICIT".equals(result);
                } else {
                    throw new SQLException("无法查询数据库supplemental logging开启状态");
                }
            }
        }
    }

    /**
     * 批量开启schema下多个表的supplemental logging
     */
    public static void enableSchemaTablesSupplementalLogging(Connection conn, String schema, Set<String> tables) {
        try {
            schema = schema.toUpperCase();
            Set<String> needEnableTables = tables.stream().map(String::toUpperCase).collect(Collectors.toSet());
            Set<String> enabledTables = getSchemaSupplementalLoggingEnabledTables(conn, schema);
            needEnableTables.removeAll(enabledTables);
            if (!needEnableTables.isEmpty()) {
                try (Statement stmt = conn.createStatement()) {
                    for (String table : needEnableTables) {
                        stmt.executeUpdate(String.format(ENABLE_TABLE_SUPPLEMENTAL_LOGGING_SQL, table));
                    }
                }
            }
        } catch (SQLException e) {
            throw new IllegalStateException("开启表supplemental logging时发生了异常", e);
        }
    }

    private static Set<String> getSchemaSupplementalLoggingEnabledTables(Connection conn, String schema) throws SQLException {
        try (Statement stmt = conn.createStatement()) {
            try (ResultSet rs = stmt.executeQuery(QUERY_TABLE_SUPPLEMENTAL_LOGGING_SQL)) {
                Set<String> tables = new HashSet<>();
                while (rs.next()) {
                    if (Arrays.asList(schema.split(",")).contains(rs.getString(1))) {
                        tables.add(rs.getString(2));
                    }
                }
                return tables;
            }
        }
    }
}
