package com.tbyf.dataadapter.console.controlle;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.tbyf.dataadapter.api.vo.ProcessorVo;
import com.tbyf.dataadapter.api.vo.Result;
import com.tbyf.dataadapter.task.processor.plugin.DataProcessorExtension;
import com.tbyf.dataadapter.task.processor.plugin.HeadProcessorExtension;
import com.tbyf.dataadapter.task.processor.plugin.ProcessorManager;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.Resource;
import org.springframework.core.io.ResourceLoader;
import org.springframework.http.ContentDisposition;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.util.ResourceUtils;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

@RestController
@RequestMapping("/plugin")
@Slf4j
public class PluginController {

    @Autowired
    ResourceLoader resourceLoader;

    @Autowired
    ProcessorManager processorManager;


    @Value("${pluginsRepoLocation}")
    private String pluginsRoot;

    @GetMapping("/download/{pluginId}")
    public ResponseEntity<?> download(@PathVariable String pluginId) {
        Path pluginPath = this.processorManager.getPluginPath(pluginId);
        if (pluginPath == null || !Files.exists(pluginPath)) {
            return ResponseEntity.notFound().build();
        }

        Resource resource = this.resourceLoader.getResource(ResourceUtils.FILE_URL_PREFIX + pluginPath);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentDisposition(
                ContentDisposition.attachment()
                        .filename(pluginPath.getFileName().toString())
                        .build());

        return ResponseEntity.ok()
                .headers(headers)
                .contentType(MediaType.APPLICATION_OCTET_STREAM)
                .body(resource);
    }

    @PostMapping("/upload")
    public Result<?> upload(MultipartFile file) {
        String filename = file.getOriginalFilename();
        try {
            Path pluginPath = Paths.get(this.pluginsRoot, filename);
            file.transferTo(pluginPath);
            processorManager.installPlugin(pluginPath.toString());
            return Result.ok(true);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.ok(false);
        }
    }

    @GetMapping("/headProcessors")
    public Result<?> getHeadProcessors() {
        Map<String, HeadProcessorExtension> processors = processorManager.getHeadProcessors();
        List<ProcessorVo> result = new ArrayList<>();
        for (String key : processors.keySet()) {
            String[] split = key.split("@");
            String pluginId = split[0];
            String processorClassname = split[1];
            String desc = processors.get(key).description();
            ProcessorVo vo = new ProcessorVo();
            vo.setPluginId(pluginId);
            vo.setProcessorClassname(processorClassname);
            vo.setDesc(desc);
            result.add(vo);
        }
        return Result.ok(result);
    }

    @GetMapping("/processors")
    public Result<?> getProcessors() {
        Map<String, DataProcessorExtension> processors = processorManager.getProcessors();
        List<ProcessorVo> result = new ArrayList<>();
        for (String key : processors.keySet()) {
            String[] split = key.split("@");
            String pluginId = split[0];
            String processorClassname = split[1];
            String desc = processors.get(key).description();
            ProcessorVo vo = new ProcessorVo();
            vo.setPluginId(pluginId);
            vo.setProcessorClassname(processorClassname);
            vo.setDesc(desc);
            result.add(vo);
        }
        return Result.ok(result);
    }
}
