package com.tbyf.dataadapter.task;

import com.tbyf.dataadapter.task.processor.*;
import com.tbyf.dataadapter.task.processor.config.PipelineDef;
import com.tbyf.dataadapter.task.processor.config.ProcessorConfig;
import com.tbyf.dataadapter.task.processor.plugin.ProcessorManager;
import com.tbyf.dataadapter.util.Utils;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;

import java.util.*;
import java.util.concurrent.ConcurrentHashMap;

@Slf4j
public class TaskManager {

    @Getter
    private final ProcessorManager processorManager;

    private final Map<String, ProcessorPipeline> pipelines = new HashMap<>();
    private final Map<String, TaskState> taskStateMap = new HashMap<>();
    private final Map<String, List<TaskThread>> taskExecutors = new HashMap<>();
    private final Map<String, Task> tasks = new ConcurrentHashMap<>();


    public TaskManager() {
        processorManager = new ProcessorManager();
        processorManager.start();
    }

    public synchronized void submitTask(TaskDef taskDef) {
        String taskId = taskDef.getTaskId();
        if (pipelines.containsKey(taskId)) {
            throw new IllegalStateException("task for id [" + taskId + "] already exists");
        }
        ProcessorPipeline pipeline = buildPipeline(taskDef.getPipelineDef());
        pipelines.put(taskId, pipeline);
        taskStateMap.put(taskId, TaskState.NEW);
        Task task = new Task();
        task.setId(taskId);
        task.setDescription(taskDef.getDescription());
        task.setState(TaskState.NEW);
        int cpuNums = Utils.getCpuNums();
        int parallelism = taskDef.getParallelism();
        if (parallelism > cpuNums) {
            log.warn("task[" + taskId + "] parallelism set too large, reset it to " + cpuNums);
            parallelism = cpuNums;
        }
        if (parallelism <= 1) {
            parallelism = 1;
        }
        task.setParallelism(parallelism);
        tasks.put(taskId, task);
    }

    private ProcessorPipeline buildPipeline(PipelineDef def) {
        ProcessorConfig head = def.getHead();
        String headId = head.getProcessorId();
        HeadProcessor headProcessor = processorManager.getHeadProcessors().get(headId);
        if (headProcessor == null) {
            throw new IllegalStateException("head processor for id [" + headId + "] not found");
        }
        headProcessor.configure(Configuration.from(head.getConfigStr()));
        DefaultProcessorPipeline pipeline = new DefaultProcessorPipeline(headProcessor);
        for (ProcessorConfig cfg : def.getProcessors()) {
            String id = cfg.getProcessorId();
            DataProcessor processor = processorManager.getProcessors().get(id);
            if (processor == null) {
                throw new IllegalStateException("processor for id [" + id + "] not found ");
            }
            processor.configure(Configuration.from(cfg.getConfigStr()));
            pipeline.addLast(processor);
        }
        return pipeline;
    }

    public synchronized void startTask(String taskId) {
        TaskState taskState = taskStateMap.get(taskId);
        if (taskState == TaskState.NEW) {
            ProcessorPipeline pipeline = pipelines.get(taskId);
            List<TaskThread> ts = new ArrayList<>();
            int parallelism = tasks.get(taskId).getParallelism();
            for (int i = 0; i < parallelism; i++) {
                TaskThread t = new TaskThread(taskId + "-" + i, pipeline::process);
                ((DefaultProcessorContext) pipeline.context()).addShutdownHook(t::terminate);
                t.addShutdownHook(() -> terminateTask(taskId));
                t.start();
                ts.add(t);
            }
            taskExecutors.put(taskId, ts);
            taskStateMap.put(taskId, TaskState.STARTED);
            updateState(taskId, TaskState.STARTED);
        }
    }

    public synchronized void pauseTask(String taskId) {
        TaskState taskState = taskStateMap.get(taskId);
        if (taskState == TaskState.STARTED) {
            taskStateMap.put(taskId, TaskState.PAUSED);
            taskExecutors.get(taskId).forEach(TaskThread::pause);
            updateState(taskId, TaskState.PAUSED);
        }
    }

    public synchronized void resumeTask(String taskId) {
        TaskState taskState = taskStateMap.get(taskId);
        if (taskState == TaskState.PAUSED) {
            taskStateMap.put(taskId, TaskState.STARTED);
            taskExecutors.get(taskId).forEach(TaskThread::proceed);
            updateState(taskId, TaskState.STARTED);
        }
    }

    public synchronized void terminateTask(String taskId) {
        TaskState taskState = taskStateMap.get(taskId);
        if (taskState == TaskState.STARTED || taskState == TaskState.PAUSED) {
            taskStateMap.put(taskId, TaskState.TERMINATED);
            taskExecutors.get(taskId).forEach(TaskThread::terminate);
            updateState(taskId, TaskState.TERMINATED);
        }
    }

    private void updateState(String taskId, TaskState state) {
        tasks.get(taskId).setState(state);
    }

    public List<Task> getTasks() {
        return new ArrayList<>(tasks.values());
    }


    public static void main(String[] args) {
        ProcessorManager.setPluginsRoot("plugins001");
        TaskManager taskManager = new TaskManager();
        TaskDef taskDef = new TaskDef();
        taskDef.setParallelism(99);
        taskDef.setTaskId("1");
//        taskDef.setParallelism(2);
        PipelineDef pDef = new PipelineDef();
        ProcessorConfig head = new ProcessorConfig();
        head.setProcessorId("new-plugin-1.1-SNAPSHOT@com.hy.example.HP");
        head.setConfigStr("uuidL=3\na=b");
        // new-plugin-1.1-SNAPSHOT@com.hy.example.DP=com.hy.example.DP
        ProcessorConfig p1 = new ProcessorConfig();
        p1.setProcessorId("new-plugin-1.1-SNAPSHOT@com.hy.example.DP");
        p1.setConfigStr("desc=描述");
        ProcessorConfig p2 = new ProcessorConfig();
        p2.setProcessorId("new-plugin-1.1-SNAPSHOT@com.hy.example.DP2");
        pDef.setHead(head);
        pDef.setProcessors(Arrays.asList(p1, p2));
        taskDef.setPipelineDef(pDef);
        taskManager.submitTask(taskDef);

        taskManager.startTask("1");
    }
}
