package com.tbyf.dataadapter.task.processor.plugin;

import com.tbyf.dataadapter.task.processor.DataProcessor;
import com.tbyf.dataadapter.task.processor.HeadProcessor;
import org.pf4j.DefaultPluginManager;
import org.pf4j.PluginWrapper;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;

public class ProcessorManager {

    private DefaultPluginManager pluginManager;

    /**
     * key: pluginId@className
     */
    private final Map<String, DataProcessorExtension> processors = new ConcurrentHashMap<>();

    private final Map<String, HeadProcessorExtension> headProcessors = new ConcurrentHashMap<>();

    public static void setPluginsRoot(String pluginsRoot) {
        System.setProperty("pf4j.pluginsDir", pluginsRoot);
    }

    public void start() {
        pluginManager = new DefaultPluginManager();
        pluginManager.loadPlugins();
        pluginManager.startPlugins();
        String pluginsRoot = System.getProperty("pf4j.pluginsDir");
        Path path = Paths.get(pluginsRoot);
        try {
            Files.createDirectories(path);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
        loadProcessors();
    }

    private void loadProcessors() {
        for (PluginWrapper plugin : pluginManager.getPlugins()) {
            String pluginId = plugin.getPluginId();

            pluginManager.getExtensions(HeadProcessorExtension.class, pluginId)
                    .forEach(hp -> {
                        hp.setPluginClassLoader(plugin.getPluginClassLoader());
                        headProcessors.put(pluginId + "@" + hp.getClass().getName(), hp);
                    });

            pluginManager.getExtensions(DataProcessorExtension.class, pluginId)
                    .stream().filter(hp -> !headProcessors.containsKey(pluginId + "@" + hp.getClass().getName()))
                    .forEach(hp -> {
                        hp.setPluginClassLoader(plugin.getPluginClassLoader());
                        processors.put(pluginId + "@" + hp.getClass().getName(), hp);
                    });
        }
    }

    public synchronized void installPlugin(String pluginPath) {
        String pluginId = pluginManager.loadPlugin(Paths.get(pluginPath));
        pluginManager.startPlugin(pluginId);
        ClassLoader cld = pluginManager.getPluginClassLoader(pluginId);

        pluginManager.getExtensions(HeadProcessorExtension.class, pluginId)
                .forEach(hp -> {
                    hp.setPluginClassLoader(cld);
                    headProcessors.put(pluginId + "@" + hp.getClass().getName(), hp);
                });

        pluginManager.getExtensions(DataProcessorExtension.class, pluginId)
                .stream().filter(hp -> !headProcessors.containsKey(pluginId + "@" + hp.getClass().getName()))
                .forEach(hp -> {
                    hp.setPluginClassLoader(cld);
                    processors.put(pluginId + "@" + hp.getClass().getName(), hp);
                });
    }

    public synchronized void deletePlugin(String pluginId) {
        pluginManager.deletePlugin(pluginId);
        headProcessors.keySet().stream()
                .filter(key -> key.startsWith(pluginId + "@")).collect(Collectors.toList())
                .forEach(headProcessors::remove);

        processors.keySet().stream()
                .filter(key -> key.startsWith(pluginId + "@")).collect(Collectors.toList())
                .forEach(processors::remove);
    }

    public synchronized String getPluginIdForPath(Path pluginPath) {
        for (PluginWrapper plugin : this.pluginManager.getPlugins()) {
            if (plugin.getPluginPath().equals(pluginPath)) {
                return plugin.getPluginId();
            }
        }
        return null;
    }

    public Map<String, DataProcessorExtension> getProcessors() {
        return new HashMap<>(processors);
    }

    public Map<String, HeadProcessorExtension> getHeadProcessors() {
        return new HashMap<>(headProcessors);
    }


    public Path getPluginPath(String pluginId) {
        return pluginManager.getPlugin(pluginId).getPluginPath();
    }
}
