package com.tbyf.dataadapter.api.impl;

import com.google.gson.reflect.TypeToken;
import com.tbyf.dataadapter.api.WorkerBizApi;
import com.tbyf.dataadapter.api.vo.ProcessorVo;
import com.tbyf.dataadapter.api.vo.Result;
import com.tbyf.dataadapter.task.Task;
import com.tbyf.dataadapter.task.TaskDef;
import com.tbyf.dataadapter.task.processor.config.PipelineDef;
import com.tbyf.dataadapter.task.processor.config.ProcessorConfig;
import com.tbyf.dataadapter.util.HttpClient;
import com.tbyf.dataadapter.util.JsonUtils;

import java.io.IOException;
import java.util.Arrays;
import java.util.List;

public class WorkerBizClient implements WorkerBizApi {

    private final String workerUrl;

    public WorkerBizClient(String workerUrl) {
        this.workerUrl = workerUrl;
    }

    @Override
    public Result<Boolean> installPlugin(String pluginId) {
        try {
            String result = HttpClient.post(workerUrl + "/installPlugin", pluginId);
            return JsonUtils.fromJson(result, new TypeToken<Result<Boolean>>() {
            });
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public Result<Boolean> submitTask(TaskDef taskDef) {
        try {
            String result = HttpClient.post(workerUrl + "/submitTask", JsonUtils.toJson(taskDef));
            return JsonUtils.fromJson(result, new TypeToken<Result<Boolean>>() {
            });
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public Result<Boolean> startTask(String taskId) {
        try {
            String result = HttpClient.post(workerUrl + "/startTask", taskId);
            return JsonUtils.fromJson(result, new TypeToken<Result<Boolean>>() {
            });
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public Result<Boolean> stopTask(String taskId) {
        try {
            String result = HttpClient.post(workerUrl + "/stopTask", taskId);
            return JsonUtils.fromJson(result, new TypeToken<Result<Boolean>>() {
            });
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public Result<List<Task>> getTasks() {
        try {
            String result = HttpClient.get(workerUrl + "/getTasks");
            return JsonUtils.fromJson(result, new TypeToken<Result<List<Task>>>() {
            });
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public Result<List<ProcessorVo>> getProcessors() {
        try {
            String result = HttpClient.get(workerUrl + "/getProcessors");
            return JsonUtils.fromJson(result, new TypeToken<Result<List<ProcessorVo>>>() {
            });
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public Result<List<ProcessorVo>> getHeadProcessors() {
        try {
            String result = HttpClient.get(workerUrl + "/getHeadProcessors");
            return JsonUtils.fromJson(result, new TypeToken<Result<List<ProcessorVo>>>() {
            });
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public static void main(String[] args) {
        WorkerBizClient client = new WorkerBizClient("http://localhost:8081");
        System.out.println(client.getHeadProcessors());
        System.out.println(client.getProcessors());
        Result<Boolean> r = client.installPlugin("new-plugin-1.1-SNAPSHOT");
        System.out.println(r);

        System.out.println(client.getProcessors());
        System.out.println(client.getHeadProcessors());

        TaskDef taskDef = new TaskDef();
        taskDef.setTaskId("02");
        taskDef.setParallelism(3);

        PipelineDef def = new PipelineDef();
        ProcessorConfig headCfg = new ProcessorConfig();
        headCfg.setProcessorId("new-plugin-1.1-SNAPSHOT@com.hy.example.HP");
        headCfg.setConfigStr("uuidL=17");
        def.setHead(headCfg);

        ProcessorConfig p1 = new ProcessorConfig();
        p1.setProcessorId("new-plugin-1.1-SNAPSHOT@com.hy.example.DP");
        p1.setConfigStr("desc=ok:");

        ProcessorConfig p2 = new ProcessorConfig();
        p2.setProcessorId("new-plugin-1.1-SNAPSHOT@com.hy.example.DP");
        p2.setConfigStr("desc=test:");

        def.setProcessors(Arrays.asList(p1, p2));
        taskDef.setPipelineDef(def);
        System.out.println(client.submitTask(taskDef));
        System.out.println(client.startTask("02"));

        client.stopTask("02");
    }
}
