package com.tbyf.dataadapter.util;

import okhttp3.*;
import okhttp3.logging.HttpLoggingInterceptor;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.concurrent.TimeUnit;

public abstract class HttpClient {

    private static final OkHttpClient client;

    public static MediaType jsonMediaType = MediaType.parse("application/json; charset=utf-8");
    public static MediaType textMediaType = MediaType.parse("application/text; charset=utf-8");

    static {
        HttpLoggingInterceptor loggingInterceptor = new HttpLoggingInterceptor();
        loggingInterceptor.level(HttpLoggingInterceptor.Level.BASIC);
        client = new OkHttpClient.Builder()
                .readTimeout(30, TimeUnit.SECONDS)
                .addInterceptor(loggingInterceptor)
                .build();
    }

    public static void download(String url, Path dest) throws IOException {
        Request request = new Request.Builder()
                .url(url)
                .build();
        try (Response response = client.newCall(request).execute()) {
            if (response.body() == null || response.body().contentLength() == 0) {
                throw new IllegalStateException("download failed");
            }
            Files.copy(response.body().byteStream(), dest);
        }
    }

    public static String post(String url, String reqBody) throws IOException {
        url = addHttpSchemaIfNeeded(url);

        Request request = new Request.Builder()
                .url(url)
                .post(RequestBody.create(reqBody, textMediaType))
                .build();
        try (Response response = client.newCall(request).execute()) {
            if (response.body() != null) {
                return response.body().string();
            } else {
                return null;
            }
        }
    }

    public static String get(String url) throws IOException {
        url = addHttpSchemaIfNeeded(url);

        Request request = new Request.Builder()
                .url(url)
                .get()
                .build();
        try (Response response = client.newCall(request).execute()) {
            if (response.body() != null) {
                return response.body().string();
            } else {
                return null;
            }
        }
    }

    private static String addHttpSchemaIfNeeded(String url) {
        if (!url.startsWith("http://") && !url.startsWith("https://")) {
            return "http://" + url;
        } else {
            return url;
        }
    }
}
