package com.tbyf.dataadapter.registry.support;

import com.tbyf.dataadapter.TimeoutException;
import com.tbyf.dataadapter.Worker;
import com.tbyf.dataadapter.registry.GroupNode;
import com.tbyf.dataadapter.registry.WorkerNode;
import com.tbyf.dataadapter.registry.WorkerRegistry;
import com.tbyf.dataadapter.util.ZkUtils;
import org.apache.curator.framework.CuratorFramework;
import org.apache.curator.framework.CuratorFrameworkFactory;
import org.apache.curator.framework.recipes.cache.ChildData;
import org.apache.curator.framework.recipes.cache.PathChildrenCache;
import org.apache.curator.framework.recipes.cache.PathChildrenCacheEvent;
import org.apache.curator.retry.ExponentialBackoffRetry;
import org.apache.curator.utils.ZKPaths;

import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;

public class ZookeeperWorkerRegistry implements WorkerRegistry {

    private final CuratorFramework zkClient;

    private final Map<String, GroupNode> groupNodes = new ConcurrentHashMap<>();

    public ZookeeperWorkerRegistry(String zkUrl) {
        CuratorFramework zkClient = CuratorFrameworkFactory.newClient(zkUrl, new ExponentialBackoffRetry(3000, 10));
        zkClient.start();
        try {
            if (!zkClient.blockUntilConnected(30, TimeUnit.SECONDS)) {
                throw new TimeoutException("zookeeper connection timeout");
            }
        } catch (Throwable e) {
            throw new RuntimeException(e);
        }
        this.zkClient = zkClient;

        try {
            init();
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    private void init() throws Exception {
        PathChildrenCache cache = new PathChildrenCache(this.zkClient, ZkPathHelper.WORKER_GROUP_ROOT_PATH, true);
        cache.getListenable().addListener((client, event) -> {
            ChildData data = event.getData();
            if (data == null) {
                return;
            }
            if (event.getType() == PathChildrenCacheEvent.Type.CHILD_ADDED) {
                String groupName = ZKPaths.getNodeFromPath(data.getPath());
                GroupNode groupNode = new GroupNode();
                groupNode.setName(groupName);
                groupNodes.put(groupName, groupNode);
                onGroupAdded(groupName);
            }
        });
        cache.start();
    }

    private void onGroupAdded(String groupName) throws Exception {
        PathChildrenCache cache = new PathChildrenCache(
                this.zkClient, ZkPathHelper.getGroupPath(groupName), true);
        cache.getListenable().addListener((client, event) -> {
            ChildData data = event.getData();
            if (data == null) {
                return;
            }
            String workerName = ZKPaths.getNodeFromPath(data.getPath());
            String workerUrl = new String(data.getData(), StandardCharsets.UTF_8);
            WorkerNode workerNode = new WorkerNode();
            workerNode.setName(workerName);
            workerNode.setAddress(workerUrl);

            switch (event.getType()) {
                case CHILD_ADDED:
                    groupNodes.get(groupName).getWorkerNodes().add(workerNode);
                    break;
                case CHILD_REMOVED:
                     groupNodes.get(groupName).getWorkerNodes().remove(workerNode);
                    break;
            }
        });
        cache.start();
    }

    @Override
    public void registerWorker(Worker worker) {
        String workerPath = ZkPathHelper.getWorkerPath(worker);
        String lockPath = ZkPathHelper.getLockPathForWorkerInGroup(worker.getGroup(), "registerWorker");
        ZkUtils.runWithinDistributedLock(zkClient, lockPath, () -> {
            if (!ZkUtils.exists(zkClient, workerPath)) {
                ZkUtils.createEphemeralZNode(zkClient, workerPath, worker.getEmbeddedServerAddress());
            } else {
                throw new IllegalStateException("worker for path [" + workerPath + "] already exists");
            }
        });
    }

    @Override
    public List<GroupNode> getWorkerGroups() {
        return new ArrayList<>(groupNodes.values());
    }

}
