package com.tbyf.dataadapter.api.impl;

import com.tbyf.dataadapter.api.WorkerBizApi;
import com.tbyf.dataadapter.api.vo.ProcessorVo;
import com.tbyf.dataadapter.api.vo.Result;
import com.tbyf.dataadapter.task.Task;
import com.tbyf.dataadapter.task.TaskDef;
import com.tbyf.dataadapter.task.TaskManager;
import com.tbyf.dataadapter.task.processor.DataProcessor;
import com.tbyf.dataadapter.task.processor.plugin.DataProcessorExtension;
import com.tbyf.dataadapter.task.processor.plugin.HeadProcessorExtension;
import com.tbyf.dataadapter.task.processor.plugin.ProcessorManager;
import com.tbyf.dataadapter.util.HttpClient;
import lombok.Setter;

import java.io.IOException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

public class WorkerBizImpl implements WorkerBizApi {

    private final TaskManager taskManager;

    private final String pluginsRoot;

    /**
     * the remote plugin repository address
     */
    private final String pluginRepository;

    public WorkerBizImpl(String pluginsRoot, String pluginRepository) {
        this.pluginsRoot = pluginsRoot;
        ProcessorManager.setPluginsRoot(pluginsRoot);
        this.taskManager = new TaskManager();
        this.pluginRepository = pluginRepository;
    }


    @Override
    public Result<Boolean> installPlugin(String pluginId) {
        Path dest = Paths.get(pluginsRoot, pluginId + ".jar");
        ProcessorManager processorManager = taskManager.getProcessorManager();
        String existingPlugin = processorManager.getPluginIdForPath(dest);
        if (existingPlugin != null) {
            processorManager.deletePlugin(existingPlugin);
        }
        try {
            HttpClient.download(getPluginDownloadUrl(pluginId), dest);
            processorManager.installPlugin(dest.toString());
            return Result.ok(true);
        } catch (IOException e) {
            return Result.ok(false);
        }
    }

    private String getPluginDownloadUrl(String pluginId) {
        return pluginRepository + "/" + pluginId;
    }

    @Override
    public Result<Boolean> submitTask(TaskDef taskDef) {
        try {
            taskManager.submitTask(taskDef);
            return Result.ok(true);
        } catch (Exception e) {
            return Result.ok(false);
        }
    }

    @Override
    public Result<Boolean> startTask(String taskId) {
        try {
            taskManager.startTask(taskId);
            return Result.ok(true);
        } catch (Exception e) {
            return Result.ok(false);
        }
    }

    @Override
    public Result<Boolean> stopTask(String taskId) {
        try {
            taskManager.terminateTask(taskId);
            return Result.ok(true);
        } catch (Exception e) {
            return Result.ok(false);
        }
    }

    @Override
    public Result<Boolean> pauseTask(String taskId) {
        try {
            taskManager.pauseTask(taskId);
            return Result.ok(true);
        } catch (Exception e) {
            return Result.ok(false);
        }
    }

    @Override
    public Result<Boolean> resumeTask(String taskId) {
        try {
            taskManager.resumeTask(taskId);
            return Result.ok(true);
        } catch (Exception e) {
            return Result.ok(false);
        }
    }

    @Override
    public Result<List<Task>> getTasks() {
        return Result.ok(taskManager.getTasks());
    }

    @Override
    public Result<List<ProcessorVo>> getProcessors() {
        ProcessorManager processorManager = taskManager.getProcessorManager();
        Map<String, DataProcessorExtension> processors = processorManager.getProcessors();
        List<ProcessorVo> result = new ArrayList<>();
        for (String key : processors.keySet()) {
            String[] split = key.split("@");
            String pluginId = split[0];
            String processorClassname = split[1];
            String desc = processors.get(key).description();
            ProcessorVo vo = new ProcessorVo();
            vo.setPluginId(pluginId);
            vo.setProcessorClassname(processorClassname);
            vo.setDesc(desc);
            result.add(vo);
        }
        return Result.ok(result);
    }

    @Override
    public Result<List<ProcessorVo>> getHeadProcessors() {
        ProcessorManager processorManager = taskManager.getProcessorManager();
        Map<String, HeadProcessorExtension> processors = processorManager.getHeadProcessors();
        List<ProcessorVo> result = new ArrayList<>();
        for (String key : processors.keySet()) {
            String[] split = key.split("@");
            String pluginId = split[0];
            String processorClassname = split[1];
            String desc = processors.get(key).description();
            ProcessorVo vo = new ProcessorVo();
            vo.setPluginId(pluginId);
            vo.setProcessorClassname(processorClassname);
            vo.setDesc(desc);
            result.add(vo);
        }
        return Result.ok(result);
    }
}
