package com.tbyf.dataadapter.http;

import com.tbyf.dataadapter.util.Assert;
import io.netty.bootstrap.ServerBootstrap;
import io.netty.buffer.Unpooled;
import io.netty.channel.*;
import io.netty.channel.nio.NioEventLoopGroup;
import io.netty.channel.socket.SocketChannel;
import io.netty.channel.socket.nio.NioServerSocketChannel;
import io.netty.handler.codec.http.*;
import io.netty.handler.timeout.IdleStateEvent;
import io.netty.handler.timeout.IdleStateHandler;
import io.netty.util.CharsetUtil;
import lombok.extern.slf4j.Slf4j;

import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;

/**
 * a simple http server used by worker
 */
@Slf4j
public class SimpleHttpServer {

    private final int port;

    private Thread t;

    private final RequestHandler requestHandler;

    public SimpleHttpServer(int port, RequestHandler requestHandler) {
        Assert.notNull(requestHandler, "request handler must not be null");
        this.port = port;
        this.requestHandler = requestHandler;
    }

    public void start() {
        t = new Thread(() -> {
            NioEventLoopGroup workerGroup = new NioEventLoopGroup();
            NioEventLoopGroup bossGroup = new NioEventLoopGroup();
            try {
                ServerBootstrap b = new ServerBootstrap();
                b.group(bossGroup, workerGroup)
                        .channel(NioServerSocketChannel.class)
                        .childHandler(new ChannelInitializer<SocketChannel>() {
                            @Override
                            protected void initChannel(SocketChannel channel) throws Exception {
                                ChannelPipeline pipeline = channel.pipeline();
                                pipeline.addLast(new IdleStateHandler(0, 0, 30, TimeUnit.SECONDS))
                                        .addLast(new HttpServerCodec())
                                        .addLast(new HttpObjectAggregator(5 * 1024 * 1024)) // 5M
                                        .addLast(new SimpleHttpRequestHandler());
                            }
                        });
                Channel ch = null;
                try {
                    ch = b.bind(port).sync().channel();
                    ch.closeFuture().sync();
                } catch (InterruptedException e) {
                    // manually stop server
                    log.info("stopping embedded server");
                    Thread.currentThread().interrupt();
                }
            } finally {
                workerGroup.shutdownGracefully();
                bossGroup.shutdownGracefully();
            }
        }, "embedded-server-main-thread");
        // t.setDaemon(true);
        t.start();
    }

    public void stop() {
        t.interrupt();
    }

    private class SimpleHttpRequestHandler extends SimpleChannelInboundHandler<FullHttpRequest> {

        @Override
        protected void channelRead0(ChannelHandlerContext context, FullHttpRequest msg) throws Exception {
            HttpRequest request = buildFromFullHttpRequest(msg);
            HttpResponse response = null;
            try {
                response = requestHandler.handle(request);
            } catch (Exception e) {
                log.error(e.getMessage(), e);
                response = HttpResponse.ERROR;
            }
            context.writeAndFlush(buildFromHttpResponse(response));
        }

        private FullHttpResponse buildFromHttpResponse(HttpResponse response) {
            response = response != null ? response : HttpResponse.NOT_FOUND;
            DefaultFullHttpResponse result = new DefaultFullHttpResponse(HttpVersion.HTTP_1_1, HttpResponseStatus.OK,
                    Unpooled.copiedBuffer(response.getContent(), CharsetUtil.UTF_8));
            result.headers().set(HttpHeaderNames.CONTENT_TYPE, response.getContentType());
            result.headers().set(HttpHeaderNames.CONTENT_LENGTH, result.content().readableBytes());
            return result;
        }


        private HttpRequest buildFromFullHttpRequest(FullHttpRequest request) {
            return HttpRequest.builder()
                    .uri(request.uri())
                    .content(request.content().toString(CharsetUtil.UTF_8))
                    .method(request.method().name())
                    .build();
        }

        @Override
        public void exceptionCaught(ChannelHandlerContext ctx, Throwable cause) throws Exception {
            ctx.close();
        }

        @Override
        public void userEventTriggered(ChannelHandlerContext ctx, Object evt) throws Exception {
            if (evt instanceof IdleStateEvent) {
                ctx.close();
            } else {
                super.userEventTriggered(ctx, evt);
            }
        }
    }

    public static void main(String[] args) {
        AtomicBoolean stopServer = new AtomicBoolean();
        SimpleHttpServer server = new SimpleHttpServer(8080, request -> {
            System.out.println(request);
            if ("/stop".equals(request.getUri())) {
                stopServer.set(true);
            }
            if ("/testErr".equals(request.getUri())) {
                throw new RuntimeException("for testing");
            }
            if ("/404".equals(request.getUri())) {
                return null;
            }
            return HttpResponse.textHtml(UUID.randomUUID().toString());
        });
        server.start();

        // todo: whether using loops is not well?
        while (!stopServer.get()) {
            try {
                Thread.sleep(500);
            } catch (InterruptedException e) {
                throw new RuntimeException(e);
            }
        }
        server.stop();
    }
}
