package com.tbyd.data.datasync.core;

import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.HashSet;
import java.util.Set;

/**
 * 开启Oracle数据库和表的supplemental logging
 */
public class CdcHelper {

    private static final String QUERY_DB_SUPPLEMENTAL_LOGGING_SQL = "SELECT SUPPLEMENTAL_LOG_DATA_MIN FROM V$DATABASE";

    private static final String ENABLE_DB_SUPPLEMENTAL_LOGGING_SQL = "ALTER DATABASE ADD SUPPLEMENTAL LOG DATA";

    private static final String QUERY_TABLE_SUPPLEMENTAL_LOGGING_SQL = "SELECT OWNER, TABLE_NAME FROM ALL_LOG_GROUPS WHERE LOG_GROUP_TYPE = 'ALL COLUMN LOGGING'";

    private static final String ENABLE_TABLE_SUPPLEMENTAL_LOGGING_SQL = "ALTER TABLE %s.%s ADD SUPPLEMENTAL LOG DATA (ALL) COLUMNS";

    /**
     * 开启Oracle数据库supplemental logging
     *
     * @param conn
     * @throws SQLException
     */
    public static void enableDBSupplementalLogging(Connection conn) throws SQLException {
        if (!isDBSupplementalLoggingEnabled(conn)) {
            try (Statement stmt = conn.createStatement()) {
                stmt.executeUpdate(ENABLE_DB_SUPPLEMENTAL_LOGGING_SQL);
            }
        }
    }

    public static boolean isDBSupplementalLoggingEnabled(Connection conn) throws SQLException {
        try (Statement stmt = conn.createStatement()) {
            try (ResultSet rs = stmt.executeQuery(QUERY_DB_SUPPLEMENTAL_LOGGING_SQL)) {
                if (rs.next()) {
                    String result = rs.getString(1);
                    return "YES".equals(result) || "IMPLICIT".equals(result);
                } else {
                    throw new SQLException("无法查询数据库的supplemental logging");
                }
            }
        }
    }

    /**
     * 开启Oracle schema下指定表的supplemental logging
     * @param conn
     * @param schema
     * @param tables
     * @throws SQLException
     */
    public static void enableSchemaTablesSupplementalLogging(Connection conn, String schema, Set<String> tables) throws SQLException {
        Set<String> needEnableTables = new HashSet<>(tables);
        Set<String> enabledTables = getSchemaSupplementalLoggingEnabledTables(conn, schema);
        needEnableTables.removeAll(enabledTables);
        if (!needEnableTables.isEmpty()) {
            try (Statement stmt = conn.createStatement()) {
                for (String table : needEnableTables) {
                    stmt.executeUpdate(String.format(ENABLE_TABLE_SUPPLEMENTAL_LOGGING_SQL, schema, table));
                }
            }
        }
    }

    private static Set<String> getSchemaSupplementalLoggingEnabledTables(Connection conn, String schema) throws SQLException {
        try (Statement stmt = conn.createStatement()) {
            try (ResultSet rs = stmt.executeQuery(QUERY_TABLE_SUPPLEMENTAL_LOGGING_SQL)) {
                Set<String> tables = new HashSet<>();
                while (rs.next()) {
                    if (schema.equalsIgnoreCase(rs.getString(1))) {
                        tables.add(rs.getString(2));
                    }
                }
                return tables;
            }
        }
    }

}