package com.tbyd.data.datasync.config;

import org.apache.commons.lang3.StringUtils;

import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Properties;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class DataSyncProperties {

    public static final String SOURCE_HOST_KEY = "source.host";
    public static final String SOURCE_PORT_KEY = "source.port";
    public static final String SOURCE_USER_KEY = "source.user";
    public static final String SOURCE_PASSWORD_KEY = "source.password";
    public static final String SOURCE_DBNAME_KEY = "source.dbname";

    public static final String DEST_HOST_KEY = "dest.host";
    public static final String DEST_PORT_KEY = "dest.port";
    public static final String DEST_USER_KEY = "dest.user";
    public static final String DEST_PASSWORD_KEY = "dest.password";
    public static final String DEST_DBNAME_KEY = "dest.dbname";

    /**
     * 指定需要同步的表，多个表之间用逗号,分隔
     * 可以为表指定唯一键，复合唯一键之间也用逗号,分隔
     * eg: tb1,tb2(k1,k2),tb3(k3)
     */
    public static final String SYNC_TABLES_KEY = "sync.tables";

    public static final String DEFAULT_PORT = "1433";

    private static final Pattern tablesWithKeyPattern = Pattern.compile("([^(,]+)(\\(([^\\)]*)\\))?");

    public static Properties loadAndValidate(String location) {
        Properties props = new Properties();
        try (InputStream is = Files.newInputStream(Paths.get(location))) {
            props.load(is);
        } catch (IOException e) {
            throw new IllegalStateException("加载配置文件失败", e);
        }

        if (StringUtils.isBlank(getSourceHost(props))) {
            throw new IllegalStateException("缺失配置项：" + SOURCE_HOST_KEY);
        }
        if (StringUtils.isBlank(getSourceUser(props))) {
            throw new IllegalStateException("缺失配置项：" + SOURCE_USER_KEY);
        }
        if (StringUtils.isBlank(getSourcePassword(props))) {
            throw new IllegalStateException("缺失配置项：" + SOURCE_PASSWORD_KEY);
        }
        if (StringUtils.isBlank(getSourceDbname(props))) {
            throw new IllegalStateException("缺失配置项：" + SOURCE_DBNAME_KEY);
        }

        if (StringUtils.isBlank(getDestinationHost(props))) {
            throw new IllegalStateException("缺失配置项：" + DEST_HOST_KEY);
        }
        if (StringUtils.isBlank(getDestinationUser(props))) {
            throw new IllegalStateException("缺失配置项：" + DEST_USER_KEY);
        }
        if (StringUtils.isBlank(getDestinationPassword(props))) {
            throw new IllegalStateException("缺失配置项：" + DEST_PASSWORD_KEY);
        }
        if (StringUtils.isBlank(getDestinationDbname(props))) {
            throw new IllegalStateException("缺失配置项：" + DEST_DBNAME_KEY);
        }

        if (StringUtils.isBlank(getSyncTableNamesWithKey(props))) {
            throw new IllegalStateException("缺失配置项：" + SYNC_TABLES_KEY);
        }
        return props;
    }

    public static String getSourceHost(Properties props) {
        return props.getProperty(SOURCE_HOST_KEY);
    }

    public static String getSourcePort(Properties props) {
        return props.getProperty(SOURCE_PORT_KEY, DEFAULT_PORT);
    }

    public static String getSourceUser(Properties props) {
        return props.getProperty(SOURCE_USER_KEY);
    }

    public static String getSourcePassword(Properties props) {
        return props.getProperty(SOURCE_PASSWORD_KEY);
    }

    public static String getSourceDbname(Properties props) {
        return props.getProperty(SOURCE_DBNAME_KEY);
    }

    public static String getDestinationHost(Properties props) {
        return props.getProperty(DEST_HOST_KEY);
    }

    public static String getDestinationPort(Properties props) {
        return props.getProperty(DEST_PORT_KEY, DEFAULT_PORT);
    }

    public static String getDestinationUser(Properties props) {
        return props.getProperty(DEST_USER_KEY);
    }

    public static String getDestinationPassword(Properties props) {
        return props.getProperty(DEST_PASSWORD_KEY);
    }

    public static String getDestinationDbname(Properties props) {
        return props.getProperty(DEST_DBNAME_KEY);
    }

    public static String getSyncTableNamesWithKey(Properties props) {
        return props.getProperty(SYNC_TABLES_KEY);
    }

    public static String getSyncTableNames(Properties props) {
        String tablesWithKey = getSyncTableNamesWithKey(props);
        List<String> tables = new ArrayList<>();
        Matcher matcher = tablesWithKeyPattern.matcher(tablesWithKey);
        while (matcher.find()) {
            tables.add(matcher.group(1));
        }
        return String.join(",", tables);
    }

    public static String[] getSyncTableNamesArray(Properties props) {
        String tablesWithKey = getSyncTableNamesWithKey(props);
        List<String> tables = new ArrayList<>();
        Matcher matcher = tablesWithKeyPattern.matcher(tablesWithKey);
        while (matcher.find()) {
            tables.add(matcher.group(1));
        }
        return tables.toArray(new String[0]);
    }

    public static List<TableAndKeys> getSyncTables(Properties props) {
        String tablesWithKey = getSyncTableNamesWithKey(props);
        List<TableAndKeys> result = new ArrayList<>();
        Matcher matcher = tablesWithKeyPattern.matcher(tablesWithKey);
        while (matcher.find()) {
            String table = matcher.group(1);
            String keys = matcher.group(3);
            String[] keysArr = StringUtils.split(keys, ",");
            result.add(new TableAndKeys(table, new HashSet<>(Arrays.asList(keysArr))));
        }
        return result;
    }

}
