package com.tbyd.data.datasync.core;

import lombok.extern.slf4j.Slf4j;

import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.HashMap;
import java.util.Map;

@Slf4j
public class DBUtils {

    public static final String SQL_SERVER_DRIVER_CLASS_NAME = "com.microsoft.sqlserver.jdbc.SQLServerDriver";

    static {
        try {
            Class.forName(SQL_SERVER_DRIVER_CLASS_NAME);
        } catch (ClassNotFoundException e) {
            throw new IllegalStateException("未找到sql server驱动", e);
        }
    }

    public static final String SQL_SERVER_URL_PATTERN = "jdbc:sqlserver://<host>:<port>";

    private static final String SWITCH_DB_SQL = "USE <dbName>";

    private static final String METADATA_QUERY_SQL = "SELECT * FROM ${tableName} WHERE 1 = 2";

    public static TableMetadata getTableMetadata(Connection conn, String tableName) {
        String sql = METADATA_QUERY_SQL.replace("${tableName}", tableName);
        try (Statement stmt = conn.createStatement()) {
            try (ResultSet rs = stmt.executeQuery(sql)) {
                ResultSetMetaData rsMetaData = rs.getMetaData();
                int columnCount = rsMetaData.getColumnCount();
                TableMetadata tableMetadata = new TableMetadata();
                tableMetadata.setTableName(tableName);
                tableMetadata.setColumnCount(columnCount);
                String[] columNames = new String[columnCount];
                Map<String, Integer> columnIndexes = new HashMap<>();
                for (int i = 0; i < columnCount; i++) {
                    columNames[i] = rsMetaData.getColumnName(i + 1);
                    columnIndexes.put(columNames[i], i);
                }
                tableMetadata.setColumnNames(columNames);
                tableMetadata.setColumnIndexes(columnIndexes);
                return tableMetadata;
            }
        } catch (SQLException e) {
            throw new IllegalStateException(e);
        }
    }

    public static String getSqlServerUrl(String host, String port) {
        return SQL_SERVER_URL_PATTERN.replace("<host>", host)
                .replace("<port>", port);
    }

    public static void switchDB(Statement stmt, String dbName) throws SQLException {
        String sql = SWITCH_DB_SQL.replace("<dbName>", dbName);
        stmt.execute(sql);
    }

    public static Connection getConn(String host, String port, String user, String password) throws SQLException {
        return DriverManager.getConnection(getSqlServerUrl(host, port), user, password);
    }

    public static void closeResource(Connection conn) {
        closeResource(null, null, conn);
    }

    public static void closeResource(ResultSet rs, Statement stmt, Connection conn) {
        if (rs != null) {
            try {
                rs.close();
            } catch (SQLException e) {
                log.error(e.getMessage(), e);
            }
        }
        if (stmt != null) {
            try {
                stmt.close();
            } catch (SQLException e) {
                log.error(e.getMessage(), e);
            }
        }
        if (conn != null) {
            try {
                conn.close();
            } catch (SQLException e) {
                log.error(e.getMessage(), e);
            }
        }
    }



}
