package cn.datax.service.system.controller;

import cn.datax.common.core.R;
import cn.datax.common.validate.ValidateGroupForSave;
import cn.datax.common.validate.ValidateGroupForUpdate;
import cn.datax.service.system.api.dto.UserDto;
import cn.datax.service.system.api.dto.UserPasswordDto;
import cn.datax.service.system.api.entity.UserEntity;
import cn.datax.service.system.mapstruct.UserMapper;
import cn.datax.service.system.service.UserService;
import cn.hutool.core.util.StrUtil;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import cn.datax.common.base.BaseController;

/**
 * <p>
 *  前端控制器
 * </p>
 *
 * @author yuwei
 * @since 2019-09-04
 */
@Api(tags = {"系统管理"})
@RestController
@RequestMapping("/users")
public class UserController extends BaseController {

    @Autowired
    private UserService userService;

    @Autowired
    private UserMapper userMapper;

    @ApiOperation(value = "获取用户详细信息", notes = "根据url的id来获取用户详细信息")
    @ApiImplicitParam(name = "id", value = "用户ID", required = true, dataType = "String", paramType = "path")
    @GetMapping("/{id}")
    public R getUserById(@PathVariable String id) {
        UserEntity userEntity = userService.getById(id);
        return R.ok().setData(userMapper.toVO(userEntity));
    }

    @ApiOperation(value = "用户分页查询", notes = "")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "pageNum", value = "当前页码", required = true, dataType = "int", example = "1"),
            @ApiImplicitParam(name = "pageSize", value = "分页条数", required = true, dataType = "int", example = "20"),
            @ApiImplicitParam(name = "user", value = "用户详细实体user", required = false, dataTypeClass = UserDto.class)
    })
    @GetMapping("/page")
    public R getUserPage(@RequestParam(value="pageNum", defaultValue="1") Integer pageNum,
                         @RequestParam(value="pageSize", defaultValue="20") Integer pageSize,
                         UserDto user) {
        QueryWrapper<UserEntity> queryWrapper = Wrappers.emptyWrapper();
        queryWrapper.like(StrUtil.isNotBlank(user.getUsername()), "username", user.getUsername());
        IPage<UserEntity> page = userService.page(new Page<>(pageNum, pageSize), queryWrapper);
        page.getRecords().stream().map(userMapper::toVO);
        return R.ok().setData(page);
    }

    @ApiOperation(value = "创建用户", notes = "根据user对象创建用户")
    @ApiImplicitParam(name = "user", value = "用户详细实体user", required = true, dataTypeClass = UserDto.class)
    @PostMapping()
    public R saveUser(@RequestBody @Validated({ValidateGroupForSave.class}) UserDto user) {
        userService.saveUser(user);
        return R.ok();
    }

    @ApiOperation(value = "更新用户详细信息", notes = "根据url的id来指定更新对象，并根据传过来的user信息来更新用户详细信息")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "用户ID", required = true, dataType = "String", paramType = "path"),
            @ApiImplicitParam(name = "user", value = "用户详细实体user", required = true, dataTypeClass = UserDto.class)
    })
    @PutMapping("/{id}")
    public R updateUser(@PathVariable String id, @RequestBody @Validated({ValidateGroupForUpdate.class}) UserDto user) {
        userService.updateUser(user);
        return R.ok();
    }

    @ApiOperation(value = "更新用户密码", notes = "根据url的id来指定更新对象，并根据传过来的user信息来更新用户密码")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "用户ID", required = true, dataType = "String", paramType = "path"),
            @ApiImplicitParam(name = "user", value = "用户详细实体user", required = true, dataTypeClass = UserPasswordDto.class)
    })
    @PutMapping("/{id}/password")
    public R updateUserPassword(@PathVariable String id, @RequestBody @Validated UserPasswordDto user) {
        userService.updateUserPassword(user);
        return R.ok();
    }

    @ApiOperation(value = "删除用户", notes = "根据url的id来指定删除对象")
    @ApiImplicitParam(name = "id", value = "用户ID", required = true, dataType = "String", paramType = "path")
    @DeleteMapping("/{id}")
    public R deleteUser(@PathVariable String id) {
        userService.deleteUserById(id);
        return R.ok();
    }

}

