package cn.datax.service.data.factory.service.impl;

import cn.datax.common.exception.DataException;
import cn.datax.common.utils.ThrowableUtil;
import cn.datax.service.data.factory.api.dto.DataSetDto;
import cn.datax.service.data.factory.api.dto.SqlParseDto;
import cn.datax.service.data.factory.api.entity.DataSetEntity;
import cn.datax.service.data.factory.service.DataSetService;
import cn.datax.service.data.factory.mapstruct.DataSetMapper;
import cn.datax.service.data.factory.dao.DataSetDao;
import cn.datax.common.base.BaseServiceImpl;
import lombok.extern.slf4j.Slf4j;
import net.sf.jsqlparser.JSQLParserException;
import net.sf.jsqlparser.expression.Function;
import net.sf.jsqlparser.parser.CCJSqlParserUtil;
import net.sf.jsqlparser.parser.SimpleNode;
import net.sf.jsqlparser.schema.Column;
import net.sf.jsqlparser.statement.Statement;
import net.sf.jsqlparser.statement.StatementVisitorAdapter;
import net.sf.jsqlparser.statement.select.*;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.List;

/**
 * <p>
 * 数据集信息表 服务实现类
 * </p>
 *
 * @author yuwei
 * @since 2020-03-20
 */
@Slf4j
@Service
@Transactional(propagation = Propagation.SUPPORTS, readOnly = true, rollbackFor = Exception.class)
public class DataSetServiceImpl extends BaseServiceImpl<DataSetDao, DataSetEntity> implements DataSetService {

    @Autowired
    private DataSetDao dataSetDao;

    @Autowired
    private DataSetMapper dataSetMapper;

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void saveDataSet(DataSetDto dataSetDto) {
        DataSetEntity dataSet = dataSetMapper.toEntity(dataSetDto);
        dataSetDao.insert(dataSet);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateDataSet(DataSetDto dataSetDto) {
        DataSetEntity dataSet = dataSetMapper.toEntity(dataSetDto);
        dataSetDao.updateById(dataSet);
    }

    @Override
    public DataSetEntity getDataSetById(String id) {
        DataSetEntity dataSetEntity = super.getById(id);
        return dataSetEntity;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteDataSetById(String id) {
        dataSetDao.deleteById(id);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteDataSetBatch(List<String> ids) {
        dataSetDao.deleteBatchIds(ids);
    }

    @Override
    public List<String> sqlAnalyse(SqlParseDto sqlParseDto) {
        String sql = sqlParseDto.getSqlText();
        Statement stmt;
        try {
            stmt = CCJSqlParserUtil.parse(sql);
        } catch (JSQLParserException e) {
            log.error("全局异常信息ex={}, StackTrace={}", e.getMessage(), ThrowableUtil.getStackTrace(e));
            throw new DataException("SQL语法有问题，解析出错");
        }
        List<String> cols = new ArrayList<>();
        if (stmt instanceof Select) {
            stmt.accept(new StatementVisitorAdapter() {
                @Override
                public void visit(Select select) {
                    select.getSelectBody().accept(new SelectVisitorAdapter() {
                        @Override
                        public void visit(PlainSelect plainSelect) {
                            plainSelect.getSelectItems().stream().forEach(selectItem -> {
                                selectItem.accept(new SelectItemVisitorAdapter() {
                                    @Override
                                    public void visit(SelectExpressionItem item) {
                                        String columnName;
                                        if (item.getAlias() == null) {
                                            SimpleNode node = item.getExpression().getASTNode();
                                            Object value = node.jjtGetValue();
                                            if (value instanceof Column) {
                                                columnName = ((Column) value).getColumnName();
                                            } else if (value instanceof Function) {
                                                columnName = value.toString();
                                            } else {
                                                // 增加对select 'aaa' from table; 的支持
                                                columnName = String.valueOf(value);
                                                columnName = columnName.replace("'", "");
                                                columnName = columnName.replace("\"", "");
                                                columnName = columnName.replace("`", "");
                                            }
                                        } else {
                                            columnName = item.getAlias().getName();
                                        }
                                        columnName = columnName.replace("'", "");
                                        columnName = columnName.replace("\"", "");
                                        columnName = columnName.replace("`", "");
                                        cols.add(columnName);
                                    }
                                });
                            });
                        }
                    });
                }
            });
        }
        return cols;
    }
}
