package cn.datax.common.rabbitmq.config;

import org.springframework.amqp.core.Binding;
import org.springframework.amqp.core.BindingBuilder;
import org.springframework.amqp.core.FanoutExchange;
import org.springframework.amqp.core.Queue;
import org.springframework.amqp.rabbit.connection.ConnectionFactory;
import org.springframework.amqp.rabbit.core.RabbitAdmin;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.config.BeanPostProcessor;
import org.springframework.context.annotation.Bean;

public class RabbitMqConfig implements BeanPostProcessor {

    @Autowired
    private RabbitAdmin rabbitAdmin;

    @Bean
    public RabbitAdmin rabbitAdmin(ConnectionFactory connectionFactory) {
        RabbitAdmin rabbitAdmin = new RabbitAdmin(connectionFactory);
        // 只有设置为 true，spring 才会加载 RabbitAdmin 这个类
        rabbitAdmin.setAutoStartup(true);
        return rabbitAdmin;
    }

    @Override
    public Object postProcessAfterInitialization(Object bean, String beanName) throws BeansException {
        rabbitAdmin.declareExchange(rabbitmqApiReleaseFanoutExchange());
        rabbitAdmin.declareQueue(fanoutExchangeQueueApiRelease());
        rabbitAdmin.declareExchange(rabbitmqApiCancelFanoutExchange());
        rabbitAdmin.declareQueue(fanoutExchangeQueueApiCancel());
        return null;
    }

    /**
     * 队列API发布
     * @return
     */
    @Bean
    public Queue fanoutExchangeQueueApiRelease() {
        /**
         * 1、name:    队列名称
         * 2、durable: 是否持久化
         * 3、exclusive: 是否独享、排外的。如果设置为true，定义为排他队列。则只有创建者可以使用此队列。也就是private私有的。
         * 4、autoDelete: 是否自动删除。也就是临时队列。当最后一个消费者断开连接后，会自动删除。
         * */
        return new Queue(RabbitMqConstant.FANOUT_EXCHANGE_QUEUE_TOPIC_API_RELEASE, true, false, false);
    }

    /**
     * 创建队列API发布FanoutExchange类型交换机
     * @return
     */
    @Bean
    public FanoutExchange rabbitmqApiReleaseFanoutExchange() {
        return new FanoutExchange(RabbitMqConstant.FANOUT_EXCHANGE_API_RELEASE_NAME, true, false);
    }

    /**
     * 队列API发布绑定到FanoutExchange交换机
     * @return
     */
    @Bean
    public Binding bindFanoutApiRelease() {
        return BindingBuilder.bind(fanoutExchangeQueueApiRelease()).to(rabbitmqApiReleaseFanoutExchange());
    }

    /**
     * 队列API注销
     * @return
     */
    @Bean
    public Queue fanoutExchangeQueueApiCancel() {
        /**
         * 1、name:    队列名称
         * 2、durable: 是否持久化
         * 3、exclusive: 是否独享、排外的。如果设置为true，定义为排他队列。则只有创建者可以使用此队列。也就是private私有的。
         * 4、autoDelete: 是否自动删除。也就是临时队列。当最后一个消费者断开连接后，会自动删除。
         * */
        return new Queue(RabbitMqConstant.FANOUT_EXCHANGE_QUEUE_TOPIC_API_CANCEL, true, false, false);
    }

    /**
     * 创建队列API注销FanoutExchange类型交换机
     * @return
     */
    @Bean
    public FanoutExchange rabbitmqApiCancelFanoutExchange() {
        return new FanoutExchange(RabbitMqConstant.FANOUT_EXCHANGE_API_CANCEL_NAME, true, false);
    }

    /**
     * 队列API注销绑定到FanoutExchange交换机
     * @return
     */
    @Bean
    public Binding bindFanoutApiCancel() {
        return BindingBuilder.bind(fanoutExchangeQueueApiCancel()).to(rabbitmqApiCancelFanoutExchange());
    }
}
