package cn.datax.common.log.aop;

import java.lang.reflect.Method;

import cn.datax.common.log.annotation.LogAop;
import cn.datax.common.log.async.AsyncTask;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.Signature;
import org.aspectj.lang.annotation.*;
import org.aspectj.lang.reflect.MethodSignature;

import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;

@Slf4j
@Aspect
public class LogAspect {

    @Autowired
    private AsyncTask asyncTask;

    // 配置织入点
    @Pointcut("@annotation(cn.datax.common.log.annotation.LogAop)")
    public void logPointCut() {}

    /**
     * 通知方法会在目标方法返回后执行
     *
     * @param joinPoint 切点
     */
    @AfterReturning(pointcut = "logPointCut()")
    public void doAfterReturning(JoinPoint joinPoint) {
        handleLog(joinPoint, null);
    }

    /**
     * 通知方法会将目标方法封装起来
     *
     * @param joinPoint 切点
     */
    @Around(value = "logPointCut()")
    public Object doAround(ProceedingJoinPoint joinPoint) throws Throwable {
        long startTime = System.currentTimeMillis();
        Object result = joinPoint.proceed();
        log.info("响应结果为{}", result);
        long endTime = System.currentTimeMillis();
        log.info("响应时间为{}毫秒", endTime - startTime);
        //如果这里不返回result，则目标对象实际返回值会被置为null
        return result;
    }

    /**
     * 通知方法会在目标方法抛出异常后执行
     *
     * @param joinPoint
     * @param e
     */
    @AfterThrowing(value = "logPointCut()", throwing = "e")
    public void doAfterThrowing(JoinPoint joinPoint, Exception e) {
        handleLog(joinPoint, e);
    }

    protected void handleLog(final JoinPoint joinPoint, final Exception e) {
        try {
            // 获得注解
            LogAop logAop = getAnnotationLog(joinPoint);
            if(logAop == null) {
                return;
            }
            // 设置方法名称
            String className = joinPoint.getTarget().getClass().getName();
            String methodName = joinPoint.getSignature().getName();
            log.info("[类名]:{},[方法]:{},[模块]:{},[描述]:{}", className, methodName, logAop.module(), logAop.value());
            // 异步保存数据库
            asyncTask.doTask();
        } catch (Exception exp) {
            log.error("前置通知异常信息:{}", exp.getMessage(), exp);
        }
    }

    /**
     * 是否存在注解，如果存在就获取
     */
    private LogAop getAnnotationLog(JoinPoint joinPoint) throws Exception {
        Signature signature = joinPoint.getSignature();
        MethodSignature methodSignature = (MethodSignature) signature;
        Method method = methodSignature.getMethod();
        if(method != null) {
            return method.getAnnotation(LogAop.class);
        }
        return null;
    }
}

