package cn.datax.service.data.factory.controller;

import cn.datax.common.core.JsonPage;
import cn.datax.common.core.R;
import cn.datax.common.validate.ValidateGroupForSave;
import cn.datax.common.validate.ValidateGroupForUpdate;
import cn.datax.service.data.factory.api.dto.DataSourceDto;
import cn.datax.service.data.factory.api.entity.DataSourceEntity;
import cn.datax.service.data.factory.api.query.DataSourceQuery;
import cn.datax.service.data.factory.api.vo.DataSourceVo;
import cn.datax.service.data.factory.mapstruct.DataSourceMapper;
import cn.datax.service.data.factory.service.DataSourceService;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import cn.datax.common.base.BaseController;

import java.util.List;
import java.util.stream.Collectors;

/**
 * <p>
 * 数据源信息表 前端控制器
 * </p>
 *
 * @author yuwei
 * @since 2020-03-14
 */
@Api(tags = {"数据源信息表"})
@RestController
@RequestMapping("/dataSource")
public class DataSourceController extends BaseController {

    @Autowired
    private DataSourceService dataSourceService;

    @Autowired
    private DataSourceMapper dataSourceMapper;

    /**
     * 通过ID查询信息
     *
     * @param id
     * @return
     */
    @ApiOperation(value = "获取详细信息", notes = "根据url的id来获取详细信息")
    @ApiImplicitParam(name = "id", value = "ID", required = true, dataType = "String", paramType = "path")
    @GetMapping("/{id}")
    public R getDataSourceById(@PathVariable String id) {
        DataSourceEntity dataSourceEntity = dataSourceService.getById(id);
        return R.ok().setData(dataSourceMapper.toVO(dataSourceEntity));
    }

    /**
     * 分页查询信息
     *
     * @param dataSourceQuery
     * @return
     */
    @ApiOperation(value = "分页查询", notes = "")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "dataSourceQuery", value = "查询实体dataSourceQuery", required = true, dataTypeClass = DataSourceQuery.class)
    })
    @GetMapping("/page")
    public R getRolePage(DataSourceQuery dataSourceQuery) {
        QueryWrapper<DataSourceEntity> queryWrapper = new QueryWrapper<>();
        IPage<DataSourceEntity> page = dataSourceService.page(new Page<>(dataSourceQuery.getPageNum(), dataSourceQuery.getPageSize()), queryWrapper);
        List<DataSourceVo> collect = page.getRecords().stream().map(dataSourceMapper::toVO).collect(Collectors.toList());
        JsonPage<DataSourceVo> jsonPage = new JsonPage<>(page.getCurrent(), page.getSize(), page.getTotal(), collect);
        return R.ok().setData(jsonPage);
    }

    /**
     * 添加
     * @param dataSource
     * @return
     */
    @ApiOperation(value = "添加信息", notes = "根据dataSource对象添加信息")
    @ApiImplicitParam(name = "dataSource", value = "详细实体dataSource", required = true, dataType = "DataSourceDto")
    @PostMapping()
    public R saveDataSource(@RequestBody @Validated({ValidateGroupForSave.class}) DataSourceDto dataSource) {
        dataSourceService.saveDataSource(dataSource);
        return R.ok();
    }

    /**
     * 修改
     * @param dataSource
     * @return
     */
    @ApiOperation(value = "修改信息", notes = "根据url的id来指定修改对象，并根据传过来的信息来修改详细信息")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "ID", required = true, dataType = "String", paramType = "path"),
            @ApiImplicitParam(name = "dataSource", value = "详细实体dataSource", required = true, dataType = "DataSourceDto")
    })
    @PutMapping("/{id}")
    public R updateDataSource(@PathVariable String id, @RequestBody @Validated({ValidateGroupForUpdate.class}) DataSourceDto dataSource) {
        dataSourceService.updateDataSource(dataSource);
        return R.ok();
    }

    /**
     * 删除
     * @param id
     * @return
     */
    @ApiOperation(value = "删除", notes = "根据url的id来指定删除对象")
    @ApiImplicitParam(name = "id", value = "ID", required = true, dataType = "String", paramType = "path")
    @DeleteMapping("/{id}")
    public R deleteDataSourceById(@PathVariable String id) {
        dataSourceService.deleteDataSourceById(id);
        return R.ok();
    }

    @PostMapping("/checkConnection")
    public R checkConnection(@RequestBody @Validated({ValidateGroupForSave.class}) DataSourceDto dataSource) {
        R r = dataSourceService.checkConnection(dataSource);
        return r;
    }
}
