package cn.datax.common.mybatis.aspectj;

import java.lang.reflect.Method;
import java.util.List;

import cn.datax.common.base.BaseQueryParams;
import cn.datax.common.core.DataConstant;
import cn.datax.common.core.DataRole;
import cn.datax.common.core.DataUser;
import cn.datax.common.mybatis.annotation.DataScopeAop;
import cn.datax.common.utils.SecurityUtil;
import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.util.StrUtil;
import lombok.extern.slf4j.Slf4j;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.Signature;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Before;
import org.aspectj.lang.annotation.Pointcut;
import org.aspectj.lang.reflect.MethodSignature;

/**
 * 数据过滤处理（基于注解式）
 */
@Slf4j
@Aspect
public class DataScopeAspect {

    // 配置织入点
    @Pointcut("@annotation(cn.datax.common.mybatis.annotation.DataScopeAop)")
    public void dataScopePointCut() {
    }

    @Before("dataScopePointCut()")
    public void doBefore(JoinPoint point) {
        handleDataScope(point);
    }

    protected void handleDataScope(final JoinPoint joinPoint) {
        // 获得注解
        DataScopeAop dataScope = getAnnotationLog(joinPoint);
        if (dataScope == null) {
            return;
        }
        DataUser currentUser = SecurityUtil.getDataUser();
        if (null != currentUser) {
            // 如果是超级管理员，则不过滤数据
            if (!currentUser.isAdmin()) {
                dataScopeFilter(joinPoint, currentUser, dataScope);
            }
        }
    }

    /**
     * 数据范围过滤
     *
     * @param user
     * @param dataScope
     */
    private void dataScopeFilter(JoinPoint joinPoint, DataUser user, DataScopeAop dataScope) {
        StringBuilder sqlString = new StringBuilder();
        List<DataRole> roles = user.getRoles();
        if (CollUtil.isNotEmpty(roles)){
            for (DataRole role : roles){
                Integer roleDataScope = role.getDataScope();
                if (DataConstant.DataScope.ALL.getKey().equals(roleDataScope)) {
                    sqlString = new StringBuilder();
                    break;
                } else if (DataConstant.DataScope.CUSTOM.getKey().equals(roleDataScope)) {
                    sqlString.append(StrUtil.format(
                            " OR {}.dept_id IN ( SELECT dept_id FROM sys_role_dept WHERE role_id = {} ) ",
                            dataScope.deptAlias()
                            ,role.getId()
                    ));
                } else if (DataConstant.DataScope.DEPT.getKey().equals(roleDataScope)) {
                    sqlString.append(StrUtil.format(
                            " OR {}.dept_id IN ( SELECT dept_id FROM sys_user_dept WHERE user_id = {} ) ",
                            dataScope.deptAlias()
                            ,user.getId()
                    ));
                } else if (DataConstant.DataScope.DEPTANDCHILD.getKey().equals(roleDataScope)) {
                    sqlString.append(StrUtil.format(
                            " OR {}.dept_id IN ( SELECT descendant FROM sys_dept_relation WHERE ancestor = {} )",
                            dataScope.deptAlias()
                            ,role.getId()
                    ));
                } else if (DataConstant.DataScope.SELF.getKey().equals(roleDataScope)) {
                    if (StrUtil.isNotBlank(dataScope.userAlias())) {
                        sqlString.append(StrUtil.format(" OR {}.user_id = {} ", dataScope.userAlias(), user.getId()));
                    } else {
                        // 数据权限为仅本人且没有userAlias别名不查询任何数据
                        sqlString.append(" OR 1=0 ");
                    }
                }
            }
        }
        if (StrUtil.isNotBlank(sqlString.toString())) {
            BaseQueryParams baseQueryParams = (BaseQueryParams) joinPoint.getArgs()[0];
            baseQueryParams.setDataScope(" AND (" + sqlString.substring(4) + ")");
        }
        log.info("数据范围过滤:{}", sqlString);
    }

    /**
     * 是否存在注解，如果存在就获取
     */
    private DataScopeAop getAnnotationLog(JoinPoint joinPoint) {
        Signature signature = joinPoint.getSignature();
        MethodSignature methodSignature = (MethodSignature) signature;
        Method method = methodSignature.getMethod();

        if (method != null) {
            return method.getAnnotation(DataScopeAop.class);
        }
        return null;
    }
}
