package cn.datax.common.mybatis.interceptor;

import cn.datax.common.core.DataConstant;
import cn.datax.common.core.DataRole;
import cn.datax.common.core.DataUser;
import cn.datax.common.mybatis.annotation.DataScope;
import cn.datax.common.utils.SecurityUtil;
import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.util.StrUtil;
import com.baomidou.mybatisplus.core.toolkit.PluginUtils;
import com.baomidou.mybatisplus.extension.handlers.AbstractSqlParserHandler;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.apache.ibatis.executor.statement.StatementHandler;
import org.apache.ibatis.mapping.BoundSql;
import org.apache.ibatis.mapping.MappedStatement;
import org.apache.ibatis.mapping.SqlCommandType;
import org.apache.ibatis.plugin.*;
import org.apache.ibatis.reflection.MetaObject;
import org.apache.ibatis.reflection.SystemMetaObject;

import java.sql.Connection;
import java.util.List;
import java.util.Map;
import java.util.Properties;

/**
 * mybatis 数据权限拦截器
 */
@Slf4j
@Intercepts({@Signature(type = StatementHandler.class, method = "prepare", args = {Connection.class, Integer.class})})
public class DataScopeInterceptor extends AbstractSqlParserHandler implements Interceptor {

	@Override
	@SneakyThrows
	public Object intercept(Invocation invocation) {
		StatementHandler statementHandler = (StatementHandler) PluginUtils.realTarget(invocation.getTarget());
		MetaObject metaObject = SystemMetaObject.forObject(statementHandler);
		this.sqlParser(metaObject);
		// 先判断是不是SELECT操作
		MappedStatement mappedStatement = (MappedStatement) metaObject.getValue("delegate.mappedStatement");
		if (!SqlCommandType.SELECT.equals(mappedStatement.getSqlCommandType())) {
			return invocation.proceed();
		}

		BoundSql boundSql = (BoundSql) metaObject.getValue("delegate.boundSql");
		String originalSql = boundSql.getSql();
		Object parameterObject = boundSql.getParameterObject();

		//查找参数中包含DataScope类型的参数
		DataScope dataScope = findDataScopeObject(parameterObject);

		if (dataScope == null) {
			return invocation.proceed();
		} else {
			// 获取当前的用户
			DataUser currentUser = SecurityUtil.getDataUser();
			if (null != currentUser) {
				// 如果是超级管理员，则不过滤数据
				if (!currentUser.isAdmin()) {
					dataScopeFilter(currentUser, dataScope);
//					originalSql = "select * from (" + originalSql + ") temp_data_scope where temp_data_scope." + scopeName + " in (" + join + ")";
//					metaObject.setValue("delegate.boundSql.sql", originalSql);
				}
			}
			log.info("originalSql:{}", originalSql);
			return invocation.proceed();
		}
	}

	/**
	 * 数据范围过滤
	 *
	 * @param user
	 * @param dataScope
	 */
	private void dataScopeFilter(DataUser user, DataScope dataScope) {
		StringBuilder sqlString = new StringBuilder();
		List<DataRole> roles = user.getRoles();
		if (CollUtil.isNotEmpty(roles)){
			for (DataRole role : roles){
				Integer roleDataScope = role.getDataScope();
				if (DataConstant.DataScope.ALL.getKey().equals(roleDataScope)) {
					sqlString = new StringBuilder();
					break;
				} else if (DataConstant.DataScope.CUSTOM.getKey().equals(roleDataScope)) {
					sqlString.append(StrUtil.format(
							" OR {}.dept_id IN ( SELECT dept_id FROM sys_role_dept WHERE role_id = {} ) ",
							dataScope.getDeptAlias()
							,role.getId()
					));
				} else if (DataConstant.DataScope.DEPT.getKey().equals(roleDataScope)) {
					sqlString.append(StrUtil.format(
							" OR {}.dept_id IN ( SELECT dept_id FROM sys_user_dept WHERE user_id = {} ) ",
							dataScope.getDeptAlias()
							,user.getId()
					));
				} else if (DataConstant.DataScope.DEPTANDCHILD.getKey().equals(roleDataScope)) {
					sqlString.append(StrUtil.format(
							" OR {}.dept_id IN ( SELECT descendant FROM sys_dept_relation WHERE ancestor = {} )",
							dataScope.getDeptAlias()
							,role.getId()
					));
				} else if (DataConstant.DataScope.SELF.getKey().equals(roleDataScope)) {
					if (StrUtil.isNotBlank(dataScope.getUserAlias())) {
						sqlString.append(StrUtil.format(" OR {}.user_id = {} ", dataScope.getUserAlias(), user.getId()));
					} else {
						// 数据权限为仅本人且没有userAlias别名不查询任何数据
						sqlString.append(" OR 1=0 ");
					}
				}
			}
		}
		log.info("数据范围过滤:{}", sqlString);
	}

	/**
	 * 生成拦截对象的代理
	 *
	 * @param target 目标对象
	 * @return 代理对象
	 */
	@Override
	public Object plugin(Object target) {
		if (target instanceof StatementHandler) {
			return Plugin.wrap(target, this);
		}
		return target;
	}

	/**
	 * mybatis配置的属性
	 *
	 * @param properties mybatis配置的属性
	 */
	@Override
	public void setProperties(Properties properties) {

	}

	/**
	 * 查找参数是否包括DataScope对象
	 *
	 * @param parameterObj 参数列表
	 * @return DataScope
	 */
	private DataScope findDataScopeObject(Object parameterObj) {
		if (parameterObj instanceof DataScope) {
			return (DataScope) parameterObj;
		} else if (parameterObj instanceof Map) {
			for (Object val : ((Map<?, ?>) parameterObj).values()) {
				if (val instanceof DataScope) {
					return (DataScope) val;
				}
			}
		}
		return null;
	}

}
