package cn.datax.service.data.factory.controller;

import cn.datax.common.core.JsonPage;
import cn.datax.common.core.R;
import cn.datax.common.database.DbQuery;
import cn.datax.common.database.core.PageResult;
import cn.datax.common.validate.ValidationGroups;
import cn.datax.service.data.factory.api.dto.DataSetDto;
import cn.datax.service.data.factory.api.entity.DataSetEntity;
import cn.datax.service.data.factory.api.query.DbDataQuery;
import cn.datax.service.data.factory.api.vo.DataSetVo;
import cn.datax.service.data.factory.api.query.DataSetQuery;
import cn.datax.service.data.factory.mapstruct.DataSetMapper;
import cn.datax.service.data.factory.service.DataSetService;
import cn.datax.service.data.factory.service.DataSourceService;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import cn.datax.common.base.BaseController;

import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * <p>
 * 数据集信息表 前端控制器
 * </p>
 *
 * @author yuwei
 * @since 2020-03-20
 */
@Api(tags = {"数据集信息表"})
@RestController
@RequestMapping("/dataSet")
public class DataSetController extends BaseController {

    @Autowired
    private DataSetService dataSetService;

    @Autowired
    private DataSetMapper dataSetMapper;

    @Autowired
    private DataSourceService dataSourceService;

    /**
     * 通过ID查询信息
     *
     * @param id
     * @return
     */
    @ApiOperation(value = "获取详细信息", notes = "根据url的id来获取详细信息")
    @ApiImplicitParam(name = "id", value = "ID", required = true, dataType = "String", paramType = "path")
    @GetMapping("/{id}")
    public R getDataSetById(@PathVariable String id) {
        DataSetEntity dataSetEntity = dataSetService.getById(id);
        return R.ok().setData(dataSetMapper.toVO(dataSetEntity));
    }

    /**
     * 分页查询信息
     *
     * @param dataSetQuery
     * @return
     */
    @ApiOperation(value = "分页查询", notes = "")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "dataSetQuery", value = "查询实体dataSetQuery", required = true, dataTypeClass = DataSetQuery.class)
    })
    @GetMapping("/page")
    public R getRolePage(DataSetQuery dataSetQuery) {
        QueryWrapper<DataSetEntity> queryWrapper = new QueryWrapper<>();
        IPage<DataSetEntity> page = dataSetService.page(new Page<>(dataSetQuery.getPageNum(), dataSetQuery.getPageSize()), queryWrapper);
        List<DataSetVo> collect = page.getRecords().stream().map(dataSetMapper::toVO).collect(Collectors.toList());
        JsonPage<DataSetVo> jsonPage = new JsonPage<>(page.getCurrent(), page.getSize(), page.getTotal(), collect);
        return R.ok().setData(jsonPage);
    }

    /**
     * 添加
     * @param dataSet
     * @return
     */
    @ApiOperation(value = "添加信息", notes = "根据dataSet对象添加信息")
    @ApiImplicitParam(name = "dataSet", value = "详细实体dataSet", required = true, dataType = "DataSetDto")
    @PostMapping()
    public R saveDataSet(@RequestBody @Validated({ValidationGroups.Insert.class}) DataSetDto dataSet) {
        dataSetService.saveDataSet(dataSet);
        return R.ok();
    }

    /**
     * 修改
     * @param dataSet
     * @return
     */
    @ApiOperation(value = "修改信息", notes = "根据url的id来指定修改对象，并根据传过来的信息来修改详细信息")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "ID", required = true, dataType = "String", paramType = "path"),
            @ApiImplicitParam(name = "dataSet", value = "详细实体dataSet", required = true, dataType = "DataSetDto")
    })
    @PutMapping("/{id}")
    public R updateDataSet(@PathVariable String id, @RequestBody @Validated({ValidationGroups.Update.class}) DataSetDto dataSet) {
        dataSetService.updateDataSet(dataSet);
        return R.ok();
    }

    /**
     * 删除
     * @param id
     * @return
     */
    @ApiOperation(value = "删除", notes = "根据url的id来指定删除对象")
    @ApiImplicitParam(name = "id", value = "ID", required = true, dataType = "String", paramType = "path")
    @DeleteMapping("/{id}")
    public R deleteDataSetById(@PathVariable String id) {
        dataSetService.deleteDataSetById(id);
        return R.ok();
    }

    @ApiOperation(value = "获取SQL结果", notes = "根据数据源的id来获取SQL结果")
    @ApiImplicitParam(name = "dbDataQuery", value = "详细实体dbDataQuery", required = true, dataType = "DbDataQuery")
    @GetMapping("/queryList")
    public R queryList(@RequestBody @Validated DbDataQuery dbDataQuery) {
        DbQuery dbQuery = dataSourceService.getDbQuery(dbDataQuery.getDataSourceId());
        List<Map<String, Object>> list = dbQuery.queryList(dbDataQuery.getSql());
        return R.ok().setData(list);
    }

    @ApiOperation(value = "分页获取SQL结果", notes = "根据数据源的id来分页获取SQL结果")
    @ApiImplicitParam(name = "dbDataQuery", value = "详细实体dbDataQuery", required = true, dataType = "DbDataQuery")
    @GetMapping("/queryByPage")
    public R queryByPage(@RequestBody @Validated DbDataQuery dbDataQuery) {
        DbQuery dbQuery = dataSourceService.getDbQuery(dbDataQuery.getDataSourceId());
        PageResult<Map<String, Object>> page = dbQuery.queryByPage(dbDataQuery.getSql(), dbDataQuery.getOffset(), dbDataQuery.getPageSize());
        page.setPageNum(dbDataQuery.getPageNum()).setPageSize(dbDataQuery.getPageSize());
        return R.ok().setData(page);
    }
}
