package cn.datax.service.data.factory.controller;

import cn.datax.common.core.DataConstant;
import cn.datax.common.core.JsonPage;
import cn.datax.common.core.R;
import cn.datax.common.validate.ValidationGroups;
import cn.datax.service.data.factory.api.dto.DataThemeDto;
import cn.datax.service.data.factory.api.entity.DataThemeEntity;
import cn.datax.service.data.factory.api.vo.DataThemeVo;
import cn.datax.service.data.factory.api.query.DataThemeQuery;
import cn.datax.service.data.factory.mapstruct.DataThemeMapper;
import cn.datax.service.data.factory.service.DataThemeService;
import cn.hutool.core.util.StrUtil;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import cn.datax.common.base.BaseController;

import java.util.List;
import java.util.stream.Collectors;

/**
 * <p>
 * 数据主题信息表 前端控制器
 * </p>
 *
 * @author yuwei
 * @since 2020-07-26
 */
@Api(tags = {"数据主题信息表"})
@RestController
@RequestMapping("/dataThemes")
public class DataThemeController extends BaseController {

    @Autowired
    private DataThemeService dataThemeService;

    @Autowired
    private DataThemeMapper dataThemeMapper;

    /**
     * 通过ID查询信息
     *
     * @param id
     * @return
     */
    @ApiOperation(value = "获取详细信息", notes = "根据url的id来获取详细信息")
    @ApiImplicitParam(name = "id", value = "ID", required = true, dataType = "String", paramType = "path")
    @GetMapping("/{id}")
    public R getDataThemeById(@PathVariable String id) {
        DataThemeEntity dataThemeEntity = dataThemeService.getDataThemeById(id);
        return R.ok().setData(dataThemeMapper.toVO(dataThemeEntity));
    }

    @ApiOperation(value = "获取列表", notes = "")
    @GetMapping("/list")
    public R getDataThemeList() {
        QueryWrapper<DataThemeEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("status", DataConstant.EnableState.ENABLE.getKey());
        List<DataThemeEntity> list = dataThemeService.list(queryWrapper);
        List<DataThemeVo> collect = list.stream().map(dataThemeMapper::toVO).collect(Collectors.toList());
        return R.ok().setData(collect);
    }

    /**
     * 分页查询信息
     *
     * @param dataThemeQuery
     * @return
     */
    @ApiOperation(value = "分页查询", notes = "")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "dataThemeQuery", value = "查询实体dataThemeQuery", required = true, dataTypeClass = DataThemeQuery.class)
    })
    @GetMapping("/page")
    public R getDataThemePage(DataThemeQuery dataThemeQuery) {
        QueryWrapper<DataThemeEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.like(StrUtil.isNotBlank(dataThemeQuery.getThemeName()), "theme_name", dataThemeQuery.getThemeName());
        IPage<DataThemeEntity> page = dataThemeService.page(new Page<>(dataThemeQuery.getPageNum(), dataThemeQuery.getPageSize()), queryWrapper);
        List<DataThemeVo> collect = page.getRecords().stream().map(dataThemeMapper::toVO).collect(Collectors.toList());
        JsonPage<DataThemeVo> jsonPage = new JsonPage<>(page.getCurrent(), page.getSize(), page.getTotal(), collect);
        return R.ok().setData(jsonPage);
    }

    /**
     * 添加
     * @param dataTheme
     * @return
     */
    @ApiOperation(value = "添加信息", notes = "根据dataTheme对象添加信息")
    @ApiImplicitParam(name = "dataTheme", value = "详细实体dataTheme", required = true, dataType = "DataThemeDto")
    @PostMapping()
    public R saveDataTheme(@RequestBody @Validated({ValidationGroups.Insert.class}) DataThemeDto dataTheme) {
        DataThemeEntity dataThemeEntity = dataThemeService.saveDataTheme(dataTheme);
        return R.ok().setData(dataThemeMapper.toVO(dataThemeEntity));
    }

    /**
     * 修改
     * @param dataTheme
     * @return
     */
    @ApiOperation(value = "修改信息", notes = "根据url的id来指定修改对象，并根据传过来的信息来修改详细信息")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "ID", required = true, dataType = "String", paramType = "path"),
            @ApiImplicitParam(name = "dataTheme", value = "详细实体dataTheme", required = true, dataType = "DataThemeDto")
    })
    @PutMapping("/{id}")
    public R updateDataTheme(@PathVariable String id, @RequestBody @Validated({ValidationGroups.Update.class}) DataThemeDto dataTheme) {
        DataThemeEntity dataThemeEntity = dataThemeService.updateDataTheme(dataTheme);
        return R.ok().setData(dataThemeMapper.toVO(dataThemeEntity));
    }

    /**
     * 删除
     * @param id
     * @return
     */
    @ApiOperation(value = "删除", notes = "根据url的id来指定删除对象")
    @ApiImplicitParam(name = "id", value = "ID", required = true, dataType = "String", paramType = "path")
    @DeleteMapping("/{id}")
    public R deleteDataThemeById(@PathVariable String id) {
        dataThemeService.deleteDataThemeById(id);
        return R.ok();
    }

    /**
     * 批量删除
     * @param ids
     * @return
     */
    @ApiOperation(value = "批量删除角色", notes = "根据url的ids来批量删除对象")
    @ApiImplicitParam(name = "ids", value = "ID集合", required = true, dataType = "List", paramType = "path")
    @DeleteMapping("/batch/{ids}")
    public R deleteDataThemeBatch(@PathVariable List<String> ids) {
        dataThemeService.deleteDataThemeBatch(ids);
        return R.ok();
    }
}
