package cn.datax.common.log.aop;

import java.lang.reflect.Method;

import cn.datax.common.log.annotation.LogAop;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.Signature;
import org.aspectj.lang.annotation.AfterReturning;
import org.aspectj.lang.annotation.AfterThrowing;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.aspectj.lang.reflect.MethodSignature;

import lombok.extern.slf4j.Slf4j;

@Slf4j
@Aspect
public class LogAspect {

    // 配置织入点
    @Pointcut("@annotation(cn.datax.common.log.annotation.LogAop)")
    public void logPointCut() {}

    /**
     * 前置通知 用于拦截操作
     *
     * @param joinPoint 切点
     */
    @AfterReturning(pointcut = "logPointCut()")
    public void doBefore(JoinPoint joinPoint) {
        handleLog(joinPoint, null);
    }

    /**
     * 拦截异常操作
     *
     * @param joinPoint
     * @param e
     */
    @AfterThrowing(value = "logPointCut()", throwing = "e")
    public void doAfter(JoinPoint joinPoint, Exception e) {
        handleLog(joinPoint, e);
    }

    protected void handleLog(final JoinPoint joinPoint, final Exception e) {
        try {
            // 获得注解
            LogAop logAop = getAnnotationLog(joinPoint);
            if(logAop == null) {
                return;
            }

            // 设置方法名称
            String className = joinPoint.getTarget().getClass().getName();
            String methodName = joinPoint.getSignature().getName();
            log.info("[类名]:{},[方法]:{},[模块]:{},[描述]:{}", className, methodName, logAop.module(), logAop.value());
        } catch (Exception exp) {
            log.error("前置通知异常信息:{}", exp.getMessage(), exp);
        }
    }

    /**
     * 是否存在注解，如果存在就获取
     */
    private LogAop getAnnotationLog(JoinPoint joinPoint) throws Exception {
        Signature signature = joinPoint.getSignature();
        MethodSignature methodSignature = (MethodSignature) signature;
        Method method = methodSignature.getMethod();
        if(method != null) {
            return method.getAnnotation(LogAop.class);
        }
        return null;
    }
}

