package cn.datax.service.system.service.impl;

import cn.datax.common.exception.DataException;
import cn.datax.service.system.api.dto.UserDto;
import cn.datax.service.system.api.dto.UserPasswordDto;
import cn.datax.service.system.api.entity.UserDeptEntity;
import cn.datax.service.system.api.entity.UserEntity;
import cn.datax.service.system.api.entity.UserPostEntity;
import cn.datax.service.system.api.entity.UserRoleEntity;
import cn.datax.service.system.dao.UserDao;
import cn.datax.service.system.dao.UserDeptDao;
import cn.datax.service.system.dao.UserPostDao;
import cn.datax.service.system.dao.UserRoleDao;
import cn.datax.service.system.mapstruct.UserMapper;
import cn.datax.service.system.service.UserService;
import cn.datax.common.base.BaseServiceImpl;
import cn.hutool.core.util.StrUtil;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.crypto.bcrypt.BCryptPasswordEncoder;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;
import java.util.stream.Collectors;

/**
 * <p>
 *  服务实现类
 * </p>
 *
 * @author yuwei
 * @since 2019-09-04
 */
@Service
@Transactional(propagation = Propagation.SUPPORTS, readOnly = true, rollbackFor = Exception.class)
public class UserServiceImpl extends BaseServiceImpl<UserDao, UserEntity> implements UserService {

    @Autowired
    private UserDeptDao userDeptMapper;
    @Autowired
    private UserPostDao userPostMapper;
    @Autowired
    private UserRoleDao userRoleMapper;
    @Autowired
    private UserMapper userMapper;

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void saveUser(UserDto userDto) {
        UserEntity user = userMapper.toEntity(userDto);
        String passwordEncode = new BCryptPasswordEncoder().encode(user.getPassword());
        user.setPassword(passwordEncode);
        baseMapper.insert(user);
        insertBatchRole(userDto.getRoles(), user.getId());
        insertBatchDept(userDto.getDepts(), user.getId());
        insertBatchPost(userDto.getPosts(), user.getId());
    }

    private void insertBatchPost(List<String> posts, String userId) {
        List<UserPostEntity> userPostList = posts
                .stream().map(postId -> {
                    UserPostEntity userPost = new UserPostEntity();
                    userPost.setUserId(userId);
                    userPost.setPostId(postId);
                    return userPost;
                }).collect(Collectors.toList());
        userPostMapper.insertBatch(userPostList);
    }

    private void insertBatchDept(List<String> deptss, String userId) {
        List<UserDeptEntity> userDeptList = deptss
                .stream().map(deptId -> {
                    UserDeptEntity userDept = new UserDeptEntity();
                    userDept.setUserId(userId);
                    userDept.setDeptId(deptId);
                    return userDept;
                }).collect(Collectors.toList());
        userDeptMapper.insertBatch(userDeptList);
    }

    private void insertBatchRole(List<String> roles, String userId) {
        List<UserRoleEntity> userRoleList = roles
                .stream().map(roleId -> {
                    UserRoleEntity userRole = new UserRoleEntity();
                    userRole.setUserId(userId);
                    userRole.setRoleId(roleId);
                    return userRole;
                }).collect(Collectors.toList());
        userRoleMapper.insertBatch(userRoleList);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateUser(UserDto userDto) {
        UserEntity user = userMapper.toEntity(userDto);
        baseMapper.updateById(user);
        userRoleMapper.delete(Wrappers.<UserRoleEntity>lambdaQuery()
                .eq(UserRoleEntity::getUserId, userDto.getId()));
        insertBatchRole(userDto.getRoles(), user.getId());
        userDeptMapper.delete(Wrappers.<UserDeptEntity>lambdaQuery()
                .eq(UserDeptEntity::getUserId, userDto.getId()));
        insertBatchDept(userDto.getDepts(), user.getId());
        userPostMapper.delete(Wrappers.<UserPostEntity>lambdaQuery()
                .eq(UserPostEntity::getUserId, userDto.getId()));
        insertBatchPost(userDto.getPosts(), user.getId());
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteUserById(String id) {
        baseMapper.deleteById(id);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateUserPassword(UserPasswordDto userPasswordDto) {
        UserEntity userEntity = baseMapper.selectById(userPasswordDto.getId());
        if(!StrUtil.equals(userEntity.getPassword(), new BCryptPasswordEncoder().encode(userPasswordDto.getOldPassword()))){
            throw new DataException("旧密码不正确");
        }
        // todo xml写更新密码sql
    }

}
