package cn.datax.service.data.factory.controller;

import cn.datax.common.core.DataConstant;
import cn.datax.common.core.JsonPage;
import cn.datax.common.core.R;
import cn.datax.common.database.DbQuery;
import cn.datax.common.database.core.DbColumn;
import cn.datax.common.database.core.DbTable;
import cn.datax.common.database.core.PageResult;
import cn.datax.common.validate.ValidationGroups;
import cn.datax.service.data.factory.api.dto.DataSourceDto;
import cn.datax.service.data.factory.api.entity.DataSourceEntity;
import cn.datax.service.data.factory.api.query.DataSourceQuery;
import cn.datax.service.data.factory.api.query.DbDataQuery;
import cn.datax.service.data.factory.api.vo.DataSourceVo;
import cn.datax.service.data.factory.mapstruct.DataSourceMapper;
import cn.datax.service.data.factory.service.DataSourceService;
import cn.hutool.core.util.StrUtil;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import cn.datax.common.base.BaseController;

import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * <p>
 * 数据源信息表 前端控制器
 * </p>
 *
 * @author yuwei
 * @since 2020-03-14
 */
@Api(tags = {"数据源信息表"})
@RestController
@RequestMapping("/dataSources")
public class DataSourceController extends BaseController {

    @Autowired
    private DataSourceService dataSourceService;

    @Autowired
    private DataSourceMapper dataSourceMapper;

    /**
     * 通过ID查询信息
     *
     * @param id
     * @return
     */
    @ApiOperation(value = "获取详细信息", notes = "根据url的id来获取详细信息")
    @ApiImplicitParam(name = "id", value = "ID", required = true, dataType = "String", paramType = "path")
    @GetMapping("/{id}")
    public R getDataSourceById(@PathVariable String id) {
        DataSourceEntity dataSourceEntity = dataSourceService.getDataSourceById(id);
        return R.ok().setData(dataSourceMapper.toVO(dataSourceEntity));
    }

    @ApiOperation(value = "获取列表", notes = "")
    @GetMapping("/list")
    public R getDataSourceList() {
        QueryWrapper<DataSourceEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("status", DataConstant.EnableState.ENABLE.getKey());
        List<DataSourceEntity> list = dataSourceService.list(queryWrapper);
        List<DataSourceVo> collect = list.stream().map(dataSourceMapper::toVO).collect(Collectors.toList());
        return R.ok().setData(collect);
    }

    /**
     * 分页查询信息
     *
     * @param dataSourceQuery
     * @return
     */
    @ApiOperation(value = "分页查询", notes = "")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "dataSourceQuery", value = "查询实体dataSourceQuery", required = true, dataTypeClass = DataSourceQuery.class)
    })
    @GetMapping("/page")
    public R getDataSourcePage(DataSourceQuery dataSourceQuery) {
        QueryWrapper<DataSourceEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.like(StrUtil.isNotBlank(dataSourceQuery.getSourceName()), "source_name", dataSourceQuery.getSourceName());
        queryWrapper.eq(StrUtil.isNotBlank(dataSourceQuery.getThemeId()), "theme_id", dataSourceQuery.getThemeId());
        IPage<DataSourceEntity> page = dataSourceService.page(new Page<>(dataSourceQuery.getPageNum(), dataSourceQuery.getPageSize()), queryWrapper);
        List<DataSourceVo> collect = page.getRecords().stream().map(dataSourceMapper::toVO).collect(Collectors.toList());
        JsonPage<DataSourceVo> jsonPage = new JsonPage<>(page.getCurrent(), page.getSize(), page.getTotal(), collect);
        return R.ok().setData(jsonPage);
    }

    /**
     * 添加
     * @param dataSource
     * @return
     */
    @ApiOperation(value = "添加信息", notes = "根据dataSource对象添加信息")
    @ApiImplicitParam(name = "dataSource", value = "详细实体dataSource", required = true, dataType = "DataSourceDto")
    @PostMapping()
    public R saveDataSource(@RequestBody @Validated({ValidationGroups.Insert.class}) DataSourceDto dataSource) {
        dataSourceService.saveDataSource(dataSource);
        return R.ok();
    }

    /**
     * 修改
     * @param dataSource
     * @return
     */
    @ApiOperation(value = "修改信息", notes = "根据url的id来指定修改对象，并根据传过来的信息来修改详细信息")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "ID", required = true, dataType = "String", paramType = "path"),
            @ApiImplicitParam(name = "dataSource", value = "详细实体dataSource", required = true, dataType = "DataSourceDto")
    })
    @PutMapping("/{id}")
    public R updateDataSource(@PathVariable String id, @RequestBody @Validated({ValidationGroups.Update.class}) DataSourceDto dataSource) {
        dataSourceService.updateDataSource(dataSource);
        return R.ok();
    }

    /**
     * 删除
     * @param id
     * @return
     */
    @ApiOperation(value = "删除", notes = "根据url的id来指定删除对象")
    @ApiImplicitParam(name = "id", value = "ID", required = true, dataType = "String", paramType = "path")
    @DeleteMapping("/{id}")
    public R deleteDataSourceById(@PathVariable String id) {
        dataSourceService.deleteDataSourceById(id);
        return R.ok();
    }

    @ApiOperation(value = "批量删除", notes = "根据url的ids来批量删除对象")
    @ApiImplicitParam(name = "ids", value = "ID集合", required = true, dataType = "List", paramType = "path")
    @DeleteMapping("/batch/{ids}")
    public R deleteDataSourceBatch(@PathVariable List<String> ids) {
        dataSourceService.deleteDataSourceBatch(ids);
        return R.ok();
    }

    /**
     * 检测数据库连通性
     * @param dataSource
     * @return
     */
    @ApiOperation(value = "数据库连通性", notes = "根据数据库配置信息检测数据库连通性")
    @ApiImplicitParam(name = "dataSource", value = "详细实体dataSource", required = true, dataType = "DataSourceDto")
    @PostMapping("/checkConnection")
    public R checkConnection(@RequestBody @Validated({ValidationGroups.Insert.class}) DataSourceDto dataSource) {
        DbQuery dbQuery = dataSourceService.checkConnection(dataSource);
        Boolean valid = dbQuery.valid();
        return valid ? R.ok() : R.error("数据库连接有误，请检查数据库配置是否正确");
    }

    /**
     * 数据库表
     * @param id
     * @return
     */
    @ApiOperation(value = "数据库表", notes = "根据数据源的id来获取指定数据库表")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "ID", required = true, dataType = "String", paramType = "path")
    })
    @GetMapping("/{id}/tables")
    public R getDbTables(@PathVariable String id) {
        List<DbTable> tables = dataSourceService.getDbTables(id);
        return R.ok().setData(tables);
    }

    /**
     * 数据库表结构
     * @param id
     * @return
     */
    @ApiOperation(value = "数据库表结构", notes = "根据数据源的id来获取指定数据库表的表结构")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "ID", required = true, dataType = "String", paramType = "path"),
            @ApiImplicitParam(name = "tableName", value = "数据库表", required = true, dataType = "String", paramType = "path")
    })
    @GetMapping("/{id}/{tableName}/columns")
    public R getDbTableColumns(@PathVariable String id, @PathVariable String tableName) {
        List<DbColumn> columns = dataSourceService.getDbTableColumns(id, tableName);
        return R.ok().setData(columns);
    }

    @ApiOperation(value = "获取SQL结果", notes = "根据数据源的id来获取SQL结果")
    @ApiImplicitParam(name = "dbDataQuery", value = "详细实体dbDataQuery", required = true, dataType = "DbDataQuery")
    @PostMapping("/queryList")
    public R queryList(@RequestBody @Validated DbDataQuery dbDataQuery) {
        DbQuery dbQuery = dataSourceService.getDbQuery(dbDataQuery.getDataSourceId());
        List<Map<String, Object>> list = dbQuery.queryList(dbDataQuery.getSql());
        return R.ok().setData(list);
    }

    @ApiOperation(value = "分页获取SQL结果", notes = "根据数据源的id来分页获取SQL结果")
    @ApiImplicitParam(name = "dbDataQuery", value = "详细实体dbDataQuery", required = true, dataType = "DbDataQuery")
    @PostMapping("/queryByPage")
    public R queryByPage(@RequestBody @Validated DbDataQuery dbDataQuery) {
        DbQuery dbQuery = dataSourceService.getDbQuery(dbDataQuery.getDataSourceId());
        PageResult<Map<String, Object>> page = dbQuery.queryByPage(dbDataQuery.getSql(), dbDataQuery.getOffset(), dbDataQuery.getPageSize());
        page.setPageNum(dbDataQuery.getPageNum()).setPageSize(dbDataQuery.getPageSize());
        return R.ok().setData(page);
    }
}
