package cn.datax.service.data.visual.service.impl;

import cn.datax.common.core.DataConstant;
import cn.datax.common.database.DataSourceFactory;
import cn.datax.common.database.DbQuery;
import cn.datax.common.database.constants.DbQueryProperty;
import cn.datax.common.exception.DataException;
import cn.datax.service.data.metadata.api.dto.DbSchema;
import cn.datax.service.data.metadata.api.entity.MetadataSourceEntity;
import cn.datax.service.data.metadata.api.feign.MetadataSourceServiceFeign;
import cn.datax.service.data.visual.api.dto.ChartColumnParse;
import cn.datax.service.data.visual.api.dto.ChartConfig;
import cn.datax.service.data.visual.api.entity.ChartEntity;
import cn.datax.service.data.visual.api.dto.ChartDto;
import cn.datax.service.data.visual.api.entity.DataSetEntity;
import cn.datax.service.data.visual.dao.DataSetDao;
import cn.datax.service.data.visual.service.ChartService;
import cn.datax.service.data.visual.mapstruct.ChartMapper;
import cn.datax.service.data.visual.dao.ChartDao;
import cn.datax.common.base.BaseServiceImpl;
import cn.hutool.core.date.DatePattern;
import cn.hutool.core.date.DateUtil;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;

/**
 * <p>
 * 可视化图表配置信息表 服务实现类
 * </p>
 *
 * @author yuwei
 * @since 2020-11-04
 */
@Service
@Transactional(propagation = Propagation.SUPPORTS, readOnly = true, rollbackFor = Exception.class)
public class ChartServiceImpl extends BaseServiceImpl<ChartDao, ChartEntity> implements ChartService {

    @Autowired
    private ChartDao chartDao;

    @Autowired
    private ChartMapper chartMapper;

    @Autowired
    private DataSetDao dataSetDao;

    @Autowired
    private DataSourceFactory dataSourceFactory;

    @Autowired
    private MetadataSourceServiceFeign metadataSourceServiceFeign;

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ChartEntity saveChart(ChartDto chartDto) {
        ChartEntity chart = chartMapper.toEntity(chartDto);
        chartDao.insert(chart);
        return chart;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ChartEntity updateChart(ChartDto chartDto) {
        ChartEntity chart = chartMapper.toEntity(chartDto);
        chartDao.updateById(chart);
        return chart;
    }

    @Override
    public ChartEntity getChartById(String id) {
        ChartEntity chartEntity = super.getById(id);
        return chartEntity;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteChartById(String id) {
        chartDao.deleteById(id);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteChartBatch(List<String> ids) {
        chartDao.deleteBatchIds(ids);
    }

    @Override
    public void copyChart(String id) {
        ChartEntity chartEntity = Optional.ofNullable(super.getById(id)).orElseThrow(() -> new DataException("获取失败"));
        ChartEntity copy = new ChartEntity();
        copy.setChartName(chartEntity.getChartName() + "_副本" + DateUtil.format(LocalDateTime.now(), DatePattern.PURE_DATETIME_PATTERN));
        copy.setChartThumbnail(chartEntity.getChartThumbnail());
        copy.setChartConfig(chartEntity.getChartConfig());
        copy.setStatus(DataConstant.EnableState.ENABLE.getKey());
        chartDao.insert(copy);
    }

    @Override
    public Map<String, Object> dataParser(ChartConfig chartConfig) {
        String datasetId = chartConfig.getDatasetId();
        DataSetEntity dataSetEntity = Optional.ofNullable(dataSetDao.selectById(datasetId)).orElseThrow(() -> new DataException("获取数据集失败"));
        MetadataSourceEntity metadataSourceEntity = Optional.ofNullable(metadataSourceServiceFeign.getMetadataSourceById(dataSetEntity.getSourceId())).orElseThrow(() -> new DataException("获取数据源失败"));
        DbSchema dbSchema = metadataSourceEntity.getDbSchema();
        DbQueryProperty dbQueryProperty = new DbQueryProperty(metadataSourceEntity.getDbType(), dbSchema.getHost(),
                dbSchema.getUsername(), dbSchema.getPassword(), dbSchema.getPort(), dbSchema.getDbName(), dbSchema.getSid());
        DbQuery dbQuery = Optional.ofNullable(dataSourceFactory.createDbQuery(dbQueryProperty)).orElseThrow(() -> new DataException("创建数据查询接口出错"));

        List<ChartColumnParse> rows = chartConfig.getRows();
        List<ChartColumnParse> columns = chartConfig.getColumns();
        List<ChartColumnParse> measures = chartConfig.getMeasures();
        String setSql = dataSetEntity.getSetSql();
        StringBuilder sql = new StringBuilder();
        List<ChartColumnParse> groups = new ArrayList<>();
        groups.addAll(rows);
        groups.addAll(columns);
        String groupJoining = groups.stream().map(s -> s.getCol()).collect(Collectors.joining(", ", " ", ","));
        sql.append("SELECT").append(groupJoining);
        String measureJoining = Optional.ofNullable(measures).orElse(new ArrayList<>()).stream().map(s -> new StringBuilder().append(s.getAggregateType()).append("(").append(s.getCol()).append(") AS ").append(s.getCol())).collect(Collectors.joining(", ", " ", " "));
        sql.append(measureJoining);
        sql.append("FROM (").append(setSql).append(") TEMP_VIEW ").append("GROUP BY ").append(groups.stream().map(s -> s.getCol()).collect(Collectors.joining(", ")));
        List<Map<String, Object>> data = dbQuery.queryList(sql.toString());
        Map<String, Object> map = new HashMap<>(2);
        map.put("data", data);
        map.put("sql", sql.toString());
        return map;
    }
}
