package cn.datax.service.data.visual.controller;

import cn.datax.common.core.DataConstant;
import cn.datax.common.core.JsonPage;
import cn.datax.common.core.R;
import cn.datax.common.validate.ValidationGroups;
import cn.datax.service.data.visual.api.dto.SqlParseDto;
import cn.datax.service.data.visual.api.dto.VisualDataSetDto;
import cn.datax.service.data.visual.api.entity.VisualDataSetEntity;
import cn.datax.service.data.visual.api.vo.VisualDataSetVo;
import cn.datax.service.data.visual.api.query.VisualDataSetQuery;
import cn.datax.service.data.visual.mapstruct.VisualDataSetMapper;
import cn.datax.service.data.visual.service.VisualDataSetService;
import cn.hutool.core.util.StrUtil;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import cn.datax.common.base.BaseController;

import java.util.List;
import java.util.stream.Collectors;

/**
 * <p>
 * 数据集信息表 前端控制器
 * </p>
 *
 * @author yuwei
 * @since 2020-07-31
 */
@Api(tags = {"数据集信息表"})
@RestController
@RequestMapping("/dataSets")
public class VisualDataSetController extends BaseController {

    @Autowired
    private VisualDataSetService visualDataSetService;

    @Autowired
    private VisualDataSetMapper visualDataSetMapper;

    /**
     * 通过ID查询信息
     *
     * @param id
     * @return
     */
    @ApiOperation(value = "获取详细信息", notes = "根据url的id来获取详细信息")
    @ApiImplicitParam(name = "id", value = "ID", required = true, dataType = "String", paramType = "path")
    @GetMapping("/{id}")
    public R getVisualDataSetById(@PathVariable String id) {
        VisualDataSetEntity visualDataSetEntity = visualDataSetService.getVisualDataSetById(id);
        return R.ok().setData(visualDataSetMapper.toVO(visualDataSetEntity));
    }

    @ApiOperation(value = "获取列表", notes = "")
    @GetMapping("/list")
    public R getVisualDataSetList() {
        QueryWrapper<VisualDataSetEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("status", DataConstant.EnableState.ENABLE.getKey());
        List<VisualDataSetEntity> list = visualDataSetService.list(queryWrapper);
        List<VisualDataSetVo> collect = list.stream().map(visualDataSetMapper::toVO).collect(Collectors.toList());
        return R.ok().setData(collect);
    }

    /**
     * 分页查询信息
     *
     * @param visualDataSetQuery
     * @return
     */
    @ApiOperation(value = "分页查询", notes = "")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "visualDataSetQuery", value = "查询实体visualDataSetQuery", required = true, dataTypeClass = VisualDataSetQuery.class)
    })
    @GetMapping("/page")
    public R getVisualDataSetPage(VisualDataSetQuery visualDataSetQuery) {
        QueryWrapper<VisualDataSetEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.like(StrUtil.isNotBlank(visualDataSetQuery.getSetName()), "set_name", visualDataSetQuery.getSetName());
        IPage<VisualDataSetEntity> page = visualDataSetService.page(new Page<>(visualDataSetQuery.getPageNum(), visualDataSetQuery.getPageSize()), queryWrapper);
        List<VisualDataSetVo> collect = page.getRecords().stream().map(visualDataSetMapper::toVO).collect(Collectors.toList());
        JsonPage<VisualDataSetVo> jsonPage = new JsonPage<>(page.getCurrent(), page.getSize(), page.getTotal(), collect);
        return R.ok().setData(jsonPage);
    }

    /**
     * 添加
     * @param visualDataSet
     * @return
     */
    @ApiOperation(value = "添加信息", notes = "根据visualDataSet对象添加信息")
    @ApiImplicitParam(name = "visualDataSet", value = "详细实体visualDataSet", required = true, dataType = "VisualDataSetDto")
    @PostMapping()
    public R saveVisualDataSet(@RequestBody @Validated({ValidationGroups.Insert.class}) VisualDataSetDto visualDataSet) {
        VisualDataSetEntity visualDataSetEntity = visualDataSetService.saveVisualDataSet(visualDataSet);
        return R.ok().setData(visualDataSetMapper.toVO(visualDataSetEntity));
    }

    /**
     * 修改
     * @param visualDataSet
     * @return
     */
    @ApiOperation(value = "修改信息", notes = "根据url的id来指定修改对象，并根据传过来的信息来修改详细信息")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "ID", required = true, dataType = "String", paramType = "path"),
            @ApiImplicitParam(name = "visualDataSet", value = "详细实体visualDataSet", required = true, dataType = "VisualDataSetDto")
    })
    @PutMapping("/{id}")
    public R updateVisualDataSet(@PathVariable String id, @RequestBody @Validated({ValidationGroups.Update.class}) VisualDataSetDto visualDataSet) {
        VisualDataSetEntity visualDataSetEntity = visualDataSetService.updateVisualDataSet(visualDataSet);
        return R.ok().setData(visualDataSetMapper.toVO(visualDataSetEntity));
    }

    /**
     * 删除
     * @param id
     * @return
     */
    @ApiOperation(value = "删除", notes = "根据url的id来指定删除对象")
    @ApiImplicitParam(name = "id", value = "ID", required = true, dataType = "String", paramType = "path")
    @DeleteMapping("/{id}")
    public R deleteVisualDataSetById(@PathVariable String id) {
        visualDataSetService.deleteVisualDataSetById(id);
        return R.ok();
    }

    /**
     * 批量删除
     * @param ids
     * @return
     */
    @ApiOperation(value = "批量删除角色", notes = "根据url的ids来批量删除对象")
    @ApiImplicitParam(name = "ids", value = "ID集合", required = true, dataType = "List", paramType = "path")
    @DeleteMapping("/batch/{ids}")
    public R deleteVisualDataSetBatch(@PathVariable List<String> ids) {
        visualDataSetService.deleteVisualDataSetBatch(ids);
        return R.ok();
    }

    /**
     * SQL解析
     * @param sqlParseDto
     * @return
     */
    @ApiOperation(value = "SQL解析")
    @ApiImplicitParam(name = "sqlParseDto", value = "SQL解析实体sqlParseDto", required = true, dataType = "SqlParseDto")
    @PostMapping("/sql/analyse")
    public R sqlAnalyse(@RequestBody @Validated SqlParseDto sqlParseDto) {
        List<String> list = visualDataSetService.sqlAnalyse(sqlParseDto);
        return R.ok().setData(list);
    }
}
