package com.tbyf.license.autoconfigure;

import com.tbyf.license.LicenseManager;
import org.springframework.boot.autoconfigure.AutoConfiguration;
import org.springframework.boot.autoconfigure.condition.ConditionalOnWebApplication;
import org.springframework.boot.autoconfigure.web.servlet.DispatcherServletAutoConfiguration;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.boot.web.servlet.FilterRegistrationBean;
import org.springframework.boot.web.servlet.ServletRegistrationBean;
import org.springframework.context.annotation.Bean;
import org.springframework.core.Ordered;

import javax.servlet.MultipartConfigElement;

@EnableConfigurationProperties(LicenseProperties.class)
@ConditionalOnWebApplication(type = ConditionalOnWebApplication.Type.SERVLET) // 仅在 Servlet 环境下生效
@AutoConfiguration(after = {DispatcherServletAutoConfiguration.class}) // DispatcherServlet 初始化后加载
// @ConditionalOnProperty(prefix = "license", name = "enabled", havingValue = "true")
public class LicenseAutoConfiguration {

    /**
     * 许可证管理器，用于安装和验证许可证
     */
    @Bean
    public LicenseManager licenseManager() {
        return new LicenseManager();
    }

    /**
     * 许可证验证过滤器，用于在请求到达前验证许可证
     */
    @Bean
    public FilterRegistrationBean<LicenseValidationFilter> licenseFilter(LicenseManager licenseManager) {
        FilterRegistrationBean<LicenseValidationFilter> registrationBean = new FilterRegistrationBean<>();
        LicenseValidationFilter filter = new LicenseValidationFilter();
        filter.setLicenseManager(licenseManager);
        registrationBean.setFilter(filter);
        registrationBean.setOrder(Ordered.HIGHEST_PRECEDENCE); // 最高优先级，确保先执行
        return registrationBean;
    }

    /**
     * 许可证安装Servlet，用于接收客户端上传的许可证文件并进行安装验证
     */
    @Bean
    public ServletRegistrationBean<LicenseInstallationServlet> licenseServlet(LicenseManager licenseManager,
                                                                              LicenseProperties licenseProperties) {
        LicenseInstallationServlet servlet = new LicenseInstallationServlet();
        servlet.setLicenseManager(licenseManager);
        servlet.setProperties(licenseProperties);
        ServletRegistrationBean<LicenseInstallationServlet> registrationBean = new ServletRegistrationBean<>(servlet, "/license/install");
        registrationBean.setMultipartConfig(new MultipartConfigElement("", 1024 * 1024, 1024 * 1024, 1024 * 1024));
        return registrationBean;
    }
}
