package com.tbyf.license;

import com.tbyf.license.autoconfigure.LicenseProperties;
import global.namespace.fun.io.bios.BIOS;
import global.namespace.truelicense.api.ConsumerLicenseManager;
import global.namespace.truelicense.api.License;
import global.namespace.truelicense.api.LicenseManagementContext;
import global.namespace.truelicense.api.LicenseManagementException;
import global.namespace.truelicense.api.LicenseValidationException;
import global.namespace.truelicense.core.passwd.ObfuscatedPasswordProtection;
import global.namespace.truelicense.obfuscate.ObfuscatedString;
import global.namespace.truelicense.v4.V4;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;

import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Arrays;
import java.util.Set;
import java.util.stream.Collectors;


public class LicenseManager implements InitializingBean {

    private static final String SUBJECT = "HIP";
    private static final String KEY_STORE_FILE = "public.ks";

    private static final LicenseManagementContext _managementContext = V4
            .builder()
            .validation(license -> verifyMAC(license.getExtra()))
            .subject(SUBJECT)
            .build();

    volatile ConsumerLicenseManager _manager;

    @Autowired
    LicenseProperties licenseProperties;

    private static void verifyMAC(Object MAC) throws LicenseValidationException {
        if (MAC == null ) {
            throw new LicenseValidationException(Messages.message("Invalid License"));
        }
        Set<String> MACs = Arrays.stream(MAC.toString().split(","))
                .map(String::trim)
                .collect(Collectors.toSet());
        if (!NetUtil.containsAnyMAC(MACs)) {
            throw new LicenseValidationException(Messages.message("Invalid License"));
        }
    }

    ConsumerLicenseManager manager() {
        ConsumerLicenseManager m = this._manager;
        return null != m ? m : (this._manager = newManager());
    }

    private ConsumerLicenseManager newManager() {
        return _managementContext
                .consumer()
                .encryption()
                .protection(protection(new long[]{0x9d823a2b21bef869L, 0x4f4790eaa895d3d8L}))
                .up()
                .authentication()
                .alias("standard")
                .loadFromResource(KEY_STORE_FILE)
                .storeProtection(protection(new long[]{0x10111721aeba73dcL, 0x1aba2dc58b17fdbeL, 0x126b920ac6703949L}))
                .up()
                .storeInUserPreferences(getClass())
                .build();
    }

    private static ObfuscatedPasswordProtection protection(long[] longs) {
        return new ObfuscatedPasswordProtection(new ObfuscatedString(longs));
    }

    public void verify() throws Exception {
        manager().verify();
    }

    public void install(String licensePath) throws Exception {
        manager().install(BIOS.file(licensePath));
    }

    public void uninstall() {
        try {
            manager().uninstall();
        }
        catch (Exception ignored) {
        }
    }

    public LicenseValidPeriod getLicenseValidPeriod() throws LicenseManagementException {
        License license = manager().load();
        LicenseValidPeriod period = new LicenseValidPeriod();
        period.setStartTime(license.getNotBefore());
        period.setExpirationTime(license.getNotAfter());
        return period;
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        Path dir = Paths.get(this.licenseProperties.getLocation()).toAbsolutePath().normalize();
        Files.createDirectories(dir);
        Path licensePath = dir.resolve(Constants.LICENSE_FILE_NAME);
        if (Files.isRegularFile(licensePath) && Files.exists(licensePath)) {
            install(licensePath.toString());
        }
    }
}
