package com.tbyf.license;

import com.tbyf.license.autoconfigure.LicenseProperties;
import global.namespace.fun.io.bios.BIOS;
import global.namespace.truelicense.api.ConsumerLicenseManager;
import global.namespace.truelicense.api.License;
import global.namespace.truelicense.api.LicenseManagementContext;
import global.namespace.truelicense.api.LicenseManagementException;
import global.namespace.truelicense.api.LicenseValidationException;
import global.namespace.truelicense.core.passwd.ObfuscatedPasswordProtection;
import global.namespace.truelicense.obfuscate.ObfuscatedString;
import global.namespace.truelicense.v4.V4;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;

import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Arrays;
import java.util.Set;
import java.util.stream.Collectors;

/**
 *  Java 软件许可证管理器
 */
public class LicenseManager implements InitializingBean {

    private static final String SUBJECT = "HIP";
    private static final String KEY_STORE_FILE = "public.ks";

    /**
     * 配置许可证管理上下文（V4版本），设置验证逻辑和主题(HIP)
     */
    private static final LicenseManagementContext _managementContext = V4
            .builder()
            .validation(license -> verifyMAC(license.getExtra()))
            .subject(SUBJECT)
            .build();

    /**
     * 消费者许可证管理器（线程安全），延迟初始化
     */
    volatile ConsumerLicenseManager _manager;

    /**
     * 从Spring配置注入的许可证存储路径等参数
     */
    @Autowired
    LicenseProperties licenseProperties;

    /**
     * 验证许可证中的MAC地址是否与当前系统匹配
     * 确保许可证仅允许在特定MAC地址的设备上运行。
     *
     * 从许可证的extra字段提取允许的MAC地址集合。
     *
     * @param MAC 许可证中包含的MAC地址
     * @throws LicenseValidationException 许可证验证异常
     */
    private static void verifyMAC(Object MAC) throws LicenseValidationException {
        if (MAC == null ) {
            throw new LicenseValidationException(Messages.message("Invalid License"));
        }
        Set<String> MACs = Arrays.stream(MAC.toString().split(","))
                .map(String::trim)
                .collect(Collectors.toSet());
        if (!NetUtil.containsAnyMAC(MACs)) {
            throw new LicenseValidationException(Messages.message("Invalid License"));
        }
    }

    ConsumerLicenseManager manager() {
        ConsumerLicenseManager m = this._manager;
        return null != m ? m : (this._manager = newManager());
    }

    private ConsumerLicenseManager newManager() {
        return _managementContext
                .consumer()
                .encryption()
                .protection(protection(new long[]{0x9d823a2b21bef869L, 0x4f4790eaa895d3d8L})) // 混淆的密钥
                .up()
                .authentication()
                .alias("standard")
                .loadFromResource(KEY_STORE_FILE) // 从类路径加载密钥库(public.ks)
                .storeProtection(protection(new long[]{0x10111721aeba73dcL, 0x1aba2dc58b17fdbeL, 0x126b920ac6703949L})) // 密钥库保护密码
                .up()
                .storeInUserPreferences(getClass())   // 存储许可证到用户偏好目录
                .build();
    }

    private static ObfuscatedPasswordProtection protection(long[] longs) {
        return new ObfuscatedPasswordProtection(new ObfuscatedString(longs));
    }

    /**
     * 许可证验证
     * 验证当前系统是否已安装有效的许可证
     *
     * @throws Exception 许可证验证异常
     */
    public void verify() throws Exception {
        manager().verify();
    }

    /**
     * 许可证安装
     * 从指定路径加载许可证文件（.lic）并验证合法性
     */
    public void install(String licensePath) throws Exception {
        uninstall();
        manager().install(BIOS.file(licensePath));
    }

    /**
     * 许可证卸载
     * 从系统中移除已安装的许可证
     */
    public void uninstall() {
        try {
            manager().uninstall();
        }
        catch (Exception ignored) {
        }
    }

    /**
     * 获取许可证有效期
     *
     * @return 许可证有效期
     * @throws LicenseManagementException 许可证管理异常
     */
    public LicenseValidPeriod getLicenseValidPeriod() throws LicenseManagementException {
        License license = manager().load();
        LicenseValidPeriod period = new LicenseValidPeriod();
        period.setStartTime(license.getNotBefore());
        period.setExpirationTime(license.getNotAfter());
        return period;
    }

    /**
     * 初始化方法
     * 确保许可证目录存在，并根据配置加载或安装许可证
     *
     * @throws Exception 初始化异常
     */
    @Override
    public void afterPropertiesSet() throws Exception {
        Path dir = Paths.get(this.licenseProperties.getLocation()).toAbsolutePath().normalize();
        Files.createDirectories(dir);
        Path licensePath = dir.resolve(Constants.LICENSE_FILE_NAME);
        if (Files.isRegularFile(licensePath) && Files.exists(licensePath)) {
            install(licensePath.toString());
        }
    }
}
