package com.tbyf.oraclecdc;

import com.tbyf.oraclecdc.util.CdcHelper;
import com.tbyf.oraclecdc.util.DBUtils;
import com.tbyf.oraclecdc.util.DateUtils;
import io.debezium.embedded.Connect;
import io.debezium.engine.DebeziumEngine;
import io.debezium.engine.spi.OffsetCommitPolicy;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.IOException;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.SQLException;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Properties;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

public class OracleCdcEngine {

    private static final Logger log = LoggerFactory.getLogger(OracleCdcEngine.class);

    private volatile DebeziumEngine<?> debeziumEngine;
    private volatile ExecutorService executor;
    private volatile Properties props;
    private final CdcProperties cdcProperties;
    private volatile RecordChangeHandler recordChangeHandler;
    private volatile Thread changeConsumerThread;

    static {
        try {
            Class.forName("oracle.jdbc.driver.OracleDriver");
        } catch (ClassNotFoundException e) {
            throw new IllegalStateException("未找到Oracle jdbc驱动", e);
        }
    }

    public OracleCdcEngine(CdcProperties cdcProperties) {
        this.cdcProperties = cdcProperties;
    }


    public void init() {
        cdcProperties.validate();
        if (cdcProperties.isAutoEnableSupplementLogging()) {
            String url = DBUtils.buildOracleJdbcUrl(cdcProperties.getHost(), cdcProperties.getPort(), cdcProperties.getDbname());
            String user = cdcProperties.getUser();
            String password = cdcProperties.getPassword();
            try (Connection conn = DriverManager.getConnection(url, user, password)) {
                CdcHelper.enableDBSupplementalLogging(conn);
                CdcHelper.enableSchemaTablesSupplementalLogging(conn, cdcProperties.getUser(),
                        new HashSet<>(Arrays.asList(cdcProperties.getTables().split(","))));
            } catch (SQLException ex) {
                throw new IllegalStateException("获取数据库连接失败", ex);
            }
        }
        mkParentDirs(cdcProperties.getOffsetFileLocation());
        mkParentDirs(cdcProperties.getHistoryFileLocation());
        props = cdcProperties.asDebeziumProps();
    }

    private static void mkParentDirs(String dirLocation) {
        new File(dirLocation).getParentFile().mkdirs();
    }

    public void start() {
        Properties props = this.props;
        if (props == null) {
            throw new IllegalStateException("Oracle CDC引擎未初始化");
        }
        final RecordChangeHandler recordChangeHandler = this.recordChangeHandler;
        if (recordChangeHandler == null) {
            throw new IllegalStateException("未设置RecordChangeHandler");
        }
        log.info("Oracle CDC引擎启动时间: {}, 引擎配置:{}", DateUtils.getNowAsString(), props);

        final RecordChangeAdapter changeAdapter = new RecordChangeAdapter();
        final Channel channel = new Channel(cdcProperties.getChannelCapacity());

        debeziumEngine = DebeziumEngine.create(Connect.class)
                .using(OffsetCommitPolicy.always())
                .using(this.props)
                .using((success, message, error) -> {
                    if (!success) {
                        log.error("Oracle CDC引擎启动失败: {}", message, error);
                        stop();
                    }
                })
                .notifying(changeEvent -> {
                    try {
                        RecordChange recordChange = changeAdapter.adapt(changeEvent);
                        if (recordChange != null) {
                            channel.put(recordChange);
                        }
                    } catch (Exception e) {
                        log.error("处理记录变更时发生了错误", e);
                    }
                }).build();

        executor = Executors.newSingleThreadExecutor();
        executor.submit(debeziumEngine);

        changeConsumerThread = new Thread(() -> {
            while (true) {
                List<RecordChange> recordChanges = channel.takeAll();
                if (recordChanges.isEmpty()) {
                    RecordChange take = channel.take();
                    if (take == null) {
                        break;
                    }
                    recordChanges = Collections.singletonList(take);
                }
                try {
                    recordChangeHandler.handleChanges(recordChanges);
                } catch (Exception e) {
                    log.error("处理记录变更时发生了错误", e);
                }
            }

        }, "oracle_cdc_consumer");
        changeConsumerThread.start();
    }

    public void setRecordChangeHandler(RecordChangeHandler recordChangeHandler) {
        this.recordChangeHandler = recordChangeHandler;
    }

    public void stop() {
        DebeziumEngine<?> debeziumEngine = this.debeziumEngine;
        if (debeziumEngine != null) {
            try {
                debeziumEngine.close();
            } catch (IOException e) {
                log.error(e.getMessage(), e);
            }
        }
        // help gc
        this.debeziumEngine = null;

        ExecutorService executor = this.executor;
        if (executor != null) {
            executor.shutdown();
        }
        this.executor = null;

        Thread changeConsumerThread = this.changeConsumerThread;
        if (changeConsumerThread != null && changeConsumerThread.isAlive()) {
            changeConsumerThread.interrupt();
        }
        this.changeConsumerThread = null;
    }


}
