package com.tbyf.oraclecdc;

import com.alibaba.druid.pool.DruidDataSource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.math.BigDecimal;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.util.List;
import java.util.Map;

/**
 * 号源剩余量同步
 */
public class AppointmentQuotaSynchronizer implements RecordChangeHandler {

    private static final Logger log = LoggerFactory.getLogger(AppointmentQuotaSynchronizer.class);

    private final DruidDataSource dataSource;

    private static final String sql = "UPDATE T_SHIFT_SCHEDULE SET SURPLUS_TOTAL = ? WHERE Shift_Code = ?";

    public AppointmentQuotaSynchronizer() {
        this.dataSource = new DruidDataSource();
        this.dataSource.setUrl(Config.TARGET_URL);
        this.dataSource.setUsername(Config.TARGET_USER);
        this.dataSource.setPassword(Config.TARGET_PASSWORD);
    }

    @Override
    public void handleChanges(List<RecordChange> changes) throws Exception {
        for (RecordChange change : changes) {
            if (change.getOpType() == RecordChange.OpType.UPDATE) {
                try {
                    Map<String, Object> after = change.getAfter();
                    BigDecimal ZXE = (BigDecimal) after.get("ZXE");   // 总号源量
                    BigDecimal GHRS = (BigDecimal) after.get("GHRS"); // 号源使用量

                    Map<String, Object> before = change.getBefore();
                    Object beforeZXE = before.get("ZXE");
                    Object beforeGHRS = before.get("GHRS");
                    if (beforeZXE.equals(ZXE) && beforeGHRS.equals(GHRS)) {
                        continue;
                    }

                    int remaining = ZXE.intValue() - GHRS.intValue(); // 号源剩余量
                    String id = (String) after.get("ID");
                    if (id != null && !id.trim().isEmpty()) {
                        try (Connection conn = dataSource.getConnection()) {
                            try (PreparedStatement stmt = conn.prepareStatement(sql)) {
                                stmt.setObject(1, remaining);
                                stmt.setString(2, id);
                                log.info("执行sql: {}", sql);
                                log.info("参数: SURPLUS_TOTAL[{}], Shift_Code[{}]", remaining, id);
                                stmt.executeUpdate();
                            }
                        }
                    }
                } catch (Exception e) {
                    log.error("处理记录变更发生错误, 变更记录:{}", change, e);
                }
            }
        }
    }

    @Override
    public void close() {
        dataSource.close();
    }
}
