package com.tbyf.oraclecdc;

import io.debezium.data.Envelope;
import io.debezium.data.VariableScaleDecimal;
import io.debezium.engine.ChangeEvent;
import io.debezium.time.Timestamp;
import org.apache.kafka.connect.data.Field;
import org.apache.kafka.connect.data.Schema;
import org.apache.kafka.connect.data.Struct;
import org.apache.kafka.connect.source.SourceRecord;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.util.HashMap;
import java.util.Map;

public class RecordChangeAdapter {

    private static final Logger log = LoggerFactory.getLogger(RecordChangeAdapter.class);

    /**
     * 如果返回null则表示忽略这个变更事件
     *
     * @param changeEvent
     * @return
     */
    public RecordChange adapt(ChangeEvent<SourceRecord, SourceRecord> changeEvent) {
        try {
            SourceRecord sr = changeEvent.value();
            Struct value = (Struct) sr.value();
            if (value == null) {
                return null;
            }
            Schema valueSchema = sr.valueSchema();
            if (!valueSchema.name().endsWith(".Envelope")) {
                return null;
            }


            RecordChange recordChange = new RecordChange();
            String destination = changeEvent.destination();
            String schemaTable = destination.substring(destination.indexOf(".") + 1);
            String[] split = schemaTable.split("\\.");
            recordChange.setSchema(split[0]);
            recordChange.setTable(split[1]);
            Struct before = value.getStruct(valueSchema.field(Envelope.FieldName.BEFORE).name());
            Struct after = value.getStruct(valueSchema.field(Envelope.FieldName.AFTER).name());
            String op = value.getString(valueSchema.field(Envelope.FieldName.OPERATION).name());
            Envelope.Operation operation = Envelope.Operation.forCode(op);
            if (operation == Envelope.Operation.CREATE) {
                recordChange.setOpType(RecordChange.OpType.CREATE);
            } else if (operation == Envelope.Operation.UPDATE) {
                recordChange.setOpType(RecordChange.OpType.UPDATE);
            } else if (operation == Envelope.Operation.DELETE) {
                recordChange.setOpType(RecordChange.OpType.DELETE);
            } else {
                return null;
            }
            recordChange.setBefore(resolveColumns(before));
            recordChange.setAfter(resolveColumns(after));

            return recordChange;
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return null;
        }
    }

    private Map<String, Object> resolveColumns(Struct cols) {
        if (cols == null) {
            return null;
        }
        HashMap<String, Object> result = new HashMap<>();
        Schema schema = cols.schema();
        for (Field field : schema.fields()) {
            Object value = cols.get(field);
            if (value != null) {
                if (field.schema().type() == Schema.Type.STRUCT) {
                    Struct structValue = (Struct) value;
                    if (field.schema().name().equals(VariableScaleDecimal.class.getName())) {
                        value = new BigDecimal(new BigInteger(structValue.getBytes(VariableScaleDecimal.VALUE_FIELD)),
                                structValue.getInt32(VariableScaleDecimal.SCALE_FIELD));
                    }
                }
                if (field.schema().type() == Schema.Type.INT64) {
                    Long longValue = (Long) value;
                    if (field.schema().name().equals(Timestamp.class.getName())) {
                        value = LocalDateTime.ofInstant(Instant.ofEpochMilli(longValue), ZoneOffset.UTC);
                    }
                }
                // todo 考虑其他情况
            }
            result.put(field.name(), value);
        }
        return result;
    }
}
