package com.tbyf.util;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import okhttp3.FormBody;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.Response;
import okhttp3.ResponseBody;

import java.math.BigInteger;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

public class XxlJobAdminClient {

    private static final OkHttpClient client = new OkHttpClient();
    private static final ObjectMapper objectMapper = new ObjectMapper();

    private static final String TOKEN_HEADER_NAME = "XXL_JOB_LOGIN_IDENTITY";

    private final String adminUrl;
    private final String token;

    public XxlJobAdminClient(String adminUrl, String username, String password) {
        if (!adminUrl.endsWith("/")) {
            adminUrl = adminUrl + "/";
        }
        this.adminUrl = adminUrl;
        this.token = buildToken(username, password);
    }

    private static String buildToken(String username, String password) {
        String token = "";
        try {
            HashMap<String, Object> map = new HashMap<>();
            map.put("username", username);
            map.put("password", DigestUtils.md5DigestAsHex(password.getBytes()));
            token = new BigInteger(objectMapper.writeValueAsString(map).getBytes()).toString(16);
        } catch (JsonProcessingException ignored) {
        }
        return token;
    }

    private List<Object> postForm(String path, FormBody formBody, ResponseHandler handler) throws Exception {
        Request request = new Request.Builder()
                .url(adminUrl + path)
                .post(formBody)
                .header(TOKEN_HEADER_NAME, token)
                .build();

        try (Response response = client.newCall(request).execute()) {
            ArrayList<Object> result = new ArrayList<>();
            if (handler != null) {
                handler.handler(response, result);
            }
            return result;
        }
    }

    /**
     * 新增任务
     *
     * @return 任务id, 返回null则表示创建失败
     */
    public Integer addJob(int executorId,
                          String jobDesc,
                          String author,
                          String alarmEmail,
                          String cronEx,
                          String jobHandler,
                          String param) {

        FormBody formBody = new FormBody.Builder()
                .add("jobGroup", String.valueOf(executorId))
                .add("jobDesc", jobDesc)
                .add("author", author)
                .add("alarmEmail", alarmEmail)
                .add("scheduleType", "CRON")
                .add("scheduleConf", cronEx)
                .add("misfireStrategy", "DO_NOTHING")
                .add("executorRouteStrategy", "ROUND")
                .add("executorHandler", jobHandler)
                .add("executorParam", param)
                .add("executorBlockStrategy", "SERIAL_EXECUTION")
                .add("executorTimeout", "0")
                .add("executorFailRetryCount", "0")
                .add("glueType", "BEAN")
                .add("glueRemark", "GLUE代码初始化")
                .build();

        try {
            return (Integer) postForm("jobinfo/add", formBody, (response, list) -> {
                ResponseBody responseBody = response.body();
                if (responseBody != null) {
                    Result result = objectMapper.readValue(responseBody.string(), Result.class);
                    String jobIdStr = (String) result.getContent();
                    if (jobIdStr == null) {
                        list.add(null);
                    } else {
                        try {
                            int jobId = Integer.parseInt(jobIdStr);
                            list.add(jobId);
                        } catch (NumberFormatException e) {
                            list.add(null);
                        }
                    }
                }
            }).get(0);
        } catch (Exception e) {
            return null;
        }
    }


    /**
     * 立即执行一次任务
     */
    public void trigger(int jobId, String param) {
        FormBody formBody = new FormBody.Builder()
                .add("id", String.valueOf(jobId))
                .add("executorParam", param)
                .build();
        try {
            postForm("jobinfo/trigger", formBody, null);
        } catch (Exception e) {
        }
    }

    public void start(int jobId) {
        FormBody formBody = new FormBody.Builder()
                .add("id", String.valueOf(jobId))
                .build();
        try {
            postForm("jobinfo/start", formBody, null);
        } catch (Exception e) {
        }
    }

    public void stop(int jobId) {
        FormBody formBody = new FormBody.Builder()
                .add("id", String.valueOf(jobId))
                .build();
        try {
            postForm("jobinfo/stop", formBody, null);
        } catch (Exception e) {
        }
    }

    public Integer getExecutorIdByName(String name) {
        FormBody formBody = new FormBody.Builder()
                .add("appname", name)
                .add("start", "0")
                .add("length", "10")
                .build();
        try {
            return (Integer) postForm("jobgroup/pageList", formBody, (response, list) -> {
                ResponseBody responseBody = response.body();
                if (responseBody != null) {
                    PageResult result = objectMapper.readValue(responseBody.string(), PageResult.class);
                    List<Object> data = result.getData();
                    if (data == null || data.isEmpty()) {
                        list.add(null);
                        return;
                    }
                    Integer id = (Integer) ((Map<String, Object>) data.get(0)).get("id");
                    list.add(id);
                }
            }).get(0);
        } catch (Exception e) {
            return null;
        }
    }

    public List<TriggerLog> getTriggerLogs(int jobId, String from, String to, int offset, int limit) {
        return doGetTriggerLogs(jobId, -1, from, to, offset, limit);
    }

    public List<TriggerLog> getSuccessfulTriggerLogs(int jobId, String from, String to, int offset, int limit) {
        return doGetTriggerLogs(jobId, 1, from, to, offset, limit);
    }

    public List<TriggerLog> getFailureTriggerLogs(int jobId, String from, String to, int offset, int limit) {
        return doGetTriggerLogs(jobId, 2, from, to, offset, limit);
    }

    public List<TriggerLog> getUncompletedTriggerLogs(int jobId, String from, String to, int offset, int limit) {
        return doGetTriggerLogs(jobId, 3, from, to, offset, limit);
    }

    @SuppressWarnings("unchecked")
    private List<TriggerLog> doGetTriggerLogs(
            /*int executorId,*/
            int jobId,
            int status,
            String from,
            String to,
            int offset,   // from 0
            int limit) {
        try {
            FormBody formBody = new FormBody.Builder()
                    .add("jobGroup", "0")
                    .add("jobId", String.valueOf(jobId))
                    .add("logStatus", String.valueOf(status))
                    .add("filterTime", from + " - " + to)
                    .add("start", String.valueOf(offset))
                    .add("length", String.valueOf(limit))
                    .build();

            return (List<TriggerLog>) postForm("joblog/pageList", formBody, (response, list) -> {
                ResponseBody responseBody = response.body();
                if (responseBody != null) {
                    PageResult result = objectMapper.readValue(responseBody.string(), PageResult.class);
                    List<TriggerLog> triggerLogs = result.getData().stream()
                            .map(e -> {
                                Map<String, Object> map = (Map<String, Object>) e;
                                TriggerLog triggerLog = new TriggerLog();
                                triggerLog.setTriggerId((Integer) map.get("id"));
                                triggerLog.setExecutorId((Integer) map.get("jobGroup"));
                                triggerLog.setJobId((Integer) map.get("jobId"));
                                triggerLog.setExecutorAddress((String) map.get("executorAddress"));
                                triggerLog.setExecutionParam((String) map.get("executorParam"));
                                triggerLog.setExecutorHandler((String) map.get("executorHandler"));
                                triggerLog.setRetryCount((Integer) map.get("executorFailRetryCount"));
                                triggerLog.setTriggerTime((String) map.get("triggerTime"));
                                triggerLog.setTriggerCode((Integer) map.get("triggerCode"));
                                triggerLog.setTriggerMsg((String) map.get("triggerMsg"));
                                triggerLog.setHandleTime((String) map.get("handleTime"));
                                triggerLog.setHandleCode((Integer) map.get("handleCode"));
                                triggerLog.setHandleMsg((String) map.get("handleMsg"));
                                triggerLog.setAlarmStatus((Integer) map.get("alarmStatus"));
                                return triggerLog;
                            }).collect(Collectors.toList());
                    list.add(triggerLogs);
                }
            }).get(0);
        } catch (Exception e) {
            return Collections.emptyList();
        }
    }

    @FunctionalInterface
    interface ResponseHandler {
        void handler(Response response, List<Object> list) throws Exception;
    }

    private static class Result {

        private int code;
        private String msg;
        private Object content;

        public int getCode() {
            return code;
        }

        public void setCode(int code) {
            this.code = code;
        }

        public String getMsg() {
            return msg;
        }

        public void setMsg(String msg) {
            this.msg = msg;
        }

        public Object getContent() {
            return content;
        }

        public void setContent(Object content) {
            this.content = content;
        }
    }

    public static class PageResult {
        private List<Object> data;
        private int recordsFiltered;
        private int recordsTotal;

        public List<Object> getData() {
            return data;
        }

        public void setData(List<Object> data) {
            this.data = data;
        }

        public int getRecordsFiltered() {
            return recordsFiltered;
        }

        public void setRecordsFiltered(int recordsFiltered) {
            this.recordsFiltered = recordsFiltered;
        }

        public int getRecordsTotal() {
            return recordsTotal;
        }

        public void setRecordsTotal(int recordsTotal) {
            this.recordsTotal = recordsTotal;
        }
    }
}
