package com.tbyf.his.web.controller.dataImport;

import com.alibaba.druid.pool.DruidDataSource;
import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.annotation.DbType;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.tbyf.his.common.annotation.IgnoreWebSecurity;
import com.tbyf.his.common.core.domain.AjaxResult;
import com.tbyf.his.common.core.text.StrFormatter;
import com.tbyf.his.common.exception.base.BaseException;
import com.tbyf.his.common.utils.StringUtils;
import com.tbyf.his.framework.datasource.DataSourceUtil;
import com.tbyf.his.framework.datasource.DynamicDataSourceContextHolder;
import com.tbyf.his.system.domain.SysDatasource;
import com.tbyf.his.system.service.ISysDatasourceService;
import com.tbyf.his.system.vo.SqlHandler;
import com.tbyf.his.web.dataImport.core.DiConstants;
import com.tbyf.his.web.dataImport.domain.vo.CreateFieldVO;
import com.tbyf.his.web.dataImport.domain.vo.TemplateVO;
import com.tbyf.his.web.dataImport.entity.DataTemplate;
import com.tbyf.his.web.dataImport.entity.ExcelData;
import com.tbyf.his.web.dataImport.service.DataFieldService;
import com.tbyf.his.web.dataImport.service.DataSourceService;
import com.tbyf.his.web.dataImport.service.DataTemplateService;
import com.tbyf.his.web.dataImport.service.ExcelDataService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.util.CollectionUtils;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;

/**
 * @author lzz
 * @date 2023/2/7 10:42
 */
@RestController
@Api(tags = "数据导入接口")
@RequestMapping("/data/import")
@Slf4j
public class DataImportController {

    @Autowired
    private DataTemplateService dataTemplateService;
    @Autowired
    private ISysDatasourceService sysDatasourceService;

    @Autowired
    private JdbcTemplate jdbcTemplate;

    @Autowired
    private DataFieldService dataFieldService;

    @Autowired
    private ExcelDataService excelDataService;

    @IgnoreWebSecurity
    @GetMapping("/datasource")
    @ApiOperation("查询所有数据源")
    public AjaxResult queryDatasource() {
        final List<SysDatasource> list = sysDatasourceService.selectSysDatasourceList(new SysDatasource());
        return AjaxResult.success(list);
    }

    @IgnoreWebSecurity
    @GetMapping("/datasource/table")
    @ApiOperation("获取指定数据源中的表")
    public AjaxResult getDbTable(@RequestParam String dataSourceId) {
        try {
            DataSourceService.switchDb(dataSourceId);
            final DruidDataSource dataSource = (DruidDataSource) DynamicDataSourceContextHolder.dataSourcesMap.get(dataSourceId);
            String sql;
            if (SqlHandler.isOracle(dataSource.getDriverClassName())) {
                sql = "SELECT TABLE_NAME AS VALUE,COMMENTS AS LABEL FROM USER_TAB_COMMENTS WHERE TABLE_TYPE = 'TABLE'";
            } else {
                sql = "select TABLE_NAME  AS VALUE,TABLE_COMMENT AS LABEL from information_schema.tables where table_type='BASE TABLE'";
            }
            final List<Map<String, Object>> maps = jdbcTemplate.queryForList(sql);
            return AjaxResult.success(maps.stream().filter(
                    distinctByKey(b -> b.get("VALUE"))
            ).distinct().collect(Collectors.toList()));
        } finally {
            DataSourceService.switchDefault();
        }
    }

    @IgnoreWebSecurity
    @GetMapping("/datasource/field")
    @ApiOperation("获取指定数据表中的字段")
    public AjaxResult getDbField(@RequestParam String dataSourceId, @RequestParam String tableName) {
        try {
            DataSourceUtil.switchDs(dataSourceId);
            final DruidDataSource dataSource = (DruidDataSource) DynamicDataSourceContextHolder.dataSourcesMap.get(dataSourceId);
            String sql = StrFormatter.format("SELECT UTC.COLUMN_NAME AS VALUE,UTC.DATA_TYPE AS TYPE,( SELECT  UCC.COMMENTS  FROM user_col_comments UCC WHERE UCC.COLUMN_NAME = UTC.COLUMN_NAME AND UCC.TABLE_NAME = UTC.TABLE_NAME ) AS LABEL FROM user_tab_columns UTC WHERE UTC.TABLE_NAME = '{}'", tableName);
            if (dataSource.getDriverClassName().toLowerCase().contains("mysql")) {
                sql = StrFormatter.format("SELECT column_name AS VALUE, DATA_TYPE AS TYPE, column_comment AS LABEL FROM information_schema.COLUMNS WHERE table_name = '{}'", tableName);
            }
            final List<Map<String, Object>> maps = jdbcTemplate.queryForList(sql);
            return AjaxResult.success(maps);
        } finally {
            DataSourceUtil.switchDefaultDs();
        }
    }


    public static <T> Predicate<T> distinctByKey(Function<? super T, ?> keyExtractor) {
        Map<Object, Boolean> seen = new ConcurrentHashMap<>();
        return t -> seen.putIfAbsent(keyExtractor.apply(t), Boolean.TRUE) == null;
    }


    @IgnoreWebSecurity
    @GetMapping("/table/create")
    @ApiOperation("物理表生成")
    public AjaxResult createTable(@RequestParam String tableName,
                                  @RequestParam String templateId) {
        // 默认表名全大写
        tableName = tableName.toUpperCase();
        final DataTemplate template = dataTemplateService.getById(templateId);
        if (StringUtils.isBlank(template.getDataSourceId())) {
            return AjaxResult.error("请在模板编辑界面选择对应的数据源");
        }
        List<CreateFieldVO> fieldList = dataFieldService.getCreateFields(template.getId());
        if (CollectionUtils.isEmpty(fieldList)) {
            return AjaxResult.error("没有可生成表的字段配置,请新建字段");
        }
        // 判断是否有重复字段
        final List<String> collect = fieldList.stream()
                .collect(Collectors.groupingBy(CreateFieldVO::getFieldName, Collectors.counting()))
                .entrySet()
                .stream()
                .filter(entry -> entry.getValue() > 1)
                .map(Map.Entry::getKey)
                .collect(Collectors.toList());
        if (collect.size() > 0) {
            return AjaxResult.error(StrFormatter.format("字段信息中有重复字段名:[{}]", collect
                    .stream().map(item -> {
                        final Optional<CreateFieldVO> first = fieldList.stream().filter(i -> StringUtils.equals(i.getFieldName(), item)).findFirst();
                        return first.map(CreateFieldVO::getTitle).orElse(null);
                    }).filter(StringUtils::isNotBlank).collect(Collectors.joining(","))));
        }
        try {
            final DbType dbType = dataTemplateService.getDbType(template.getDataSourceId());
            DataSourceService.switchDb(template.getDataSourceId());
            if (dbType.equals(DbType.MYSQL)) {
                // mysql
                return AjaxResult.error("mysql数据因字段数量以及长度原因,暂不支持,请联系管理员");
                /*final Integer count = jdbcTemplate.queryForObject(StrFormatter.format("select count(table_name) from information_schema.TABLES where TABLE_NAME = '{}'", tableName), Integer.class);
                if (count > 0) {
                    return AjaxResult.error("此表已存在");
                }
                StringBuilder sb = new StringBuilder();
                sb.append("create table {} ( ");
                fieldList.forEach(field -> sb.append(field.getField())
                        .append(" varchar(32) null comment '")
                        .append(field.createComment()).append("',"));
                sb.append(" YEAR varchar(32) null").append(" ) comment '")
                        .append(template.createComment())
                        .append("';");
                // 同时创建数据表与临时表
                log.info("生成物理表的sql为:{}", sb);
                jdbcTemplate.execute(StrFormatter.format(sb.toString(), tableName));
                jdbcTemplate.execute(StrFormatter.format(sb.toString(), tableName + "_TEMP"));
                // 修改模板表绑定的表名
                final DataTemplate updateTemplate = new DataTemplate();
                updateTemplate.setId(template.getId());
                updateTemplate.setTableName(tableName);
                DataSourceUtil.switchDefaultDs();
                dataImportService.updateById(updateTemplate);*/
            } else {
                // oracle
                final Integer count = jdbcTemplate.queryForObject(StrFormatter.format("select count(*) from user_tables where table_name =upper('{}')", tableName), Integer.class);
                if (count > 0) {
                    return AjaxResult.error("此表已存在");
                }
                List<String> prodSqlList = new ArrayList<>();
                List<String> tempSqlList = new ArrayList<>();
                final String tempTableName = tableName + "_TEMP";
                String comment = "COMMENT ON COLUMN {}.{} IS '{}'";
                StringBuilder sql = new StringBuilder();
                sql.append("CREATE TABLE {} ( ");
                String finalTableName = tableName;
                fieldList.forEach(field -> {
                    sql.append(field.getFieldName()).append(" ").append(field.getFieldType()).append(" ,");
                    prodSqlList.add(StrFormatter.format(comment, finalTableName, field.getFieldName(), field.getTitle()));
                    tempSqlList.add(StrFormatter.format(comment, tempTableName, field.getFieldName(), field.getTitle()));
                });
                sql.append(String.join("", DiConstants.DEFAULT_FIELD_TEMPLATE));
                sql.deleteCharAt(sql.length() - 1);
                sql.append(")");
                DiConstants.DEFAULT_FIELD_COMMENT_TEMPLATE.forEach(item -> {
                    prodSqlList.add(item.replaceAll("@tableName", finalTableName));
                    tempSqlList.add(item.replaceAll("@tableName", tempTableName));
                });
                prodSqlList.add(0, StrFormatter.format(sql.toString(), finalTableName));
                tempSqlList.add(0, StrFormatter.format(sql.toString(), tempTableName));
                prodSqlList.add(StrFormatter.format("COMMENT ON TABLE {} IS '{}'", finalTableName, template.createComment()));
                tempSqlList.add(StrFormatter.format("COMMENT ON TABLE {} IS '{}'", tempTableName, template.createComment()));
                // 同时创建数据表与临时表
                String[] prod = new String[prodSqlList.size()];
                String[] temp = new String[tempSqlList.size()];
                prodSqlList.toArray(prod);
                tempSqlList.toArray(temp);
                jdbcTemplate.batchUpdate(prod);
                jdbcTemplate.batchUpdate(temp);
                // 修改模板表绑定的表名
                final DataTemplate updateTemplate = new DataTemplate();
                updateTemplate.setId(template.getId());
                updateTemplate.setTableName(finalTableName);
                DataSourceService.switchDefault();
                dataTemplateService.updateById(updateTemplate);
            }
            return AjaxResult.success();
        } catch (Exception e) {
            log.error("物理表生成失败", e);
            return AjaxResult.error(e.getMessage());
        } finally {
            DataSourceService.switchDefault();
        }
    }

    @IgnoreWebSecurity
    @GetMapping("/analyze/export")
    @ApiOperation("数据分析并导出")
    public void analyzeExport(@RequestParam String templateId,
                              HttpServletResponse response) {
        ExcelData excelData = excelDataService.getOne(Wrappers.lambdaQuery(ExcelData.class)
                .eq(ExcelData::getTemplateId, templateId)
                .eq(ExcelData::getType, "1"), false);
        if (excelData == null) {
            try {
                response.setContentType("application/json");
                response.setCharacterEncoding("UTF-8");
                response.getWriter().println(JSON.toJSONString(AjaxResult.error("请先导入数据文件")));
                response.getWriter().flush();
            } catch (IOException e) {
                throw new RuntimeException(e);
            }
            return;
        }
        excelDataService.analyzeExport(excelData, response);
    }

    @IgnoreWebSecurity
    @PostMapping("/clearTemp")
    @ApiOperation("清空临时表数据")
    public AjaxResult clearTemp(@RequestBody TemplateVO vo) {
        final DataTemplate template = dataTemplateService.getById(vo.getId());
        try {
            DataSourceService.switchDb(template.getDataSourceId());
            String sql = StrFormatter.format("DELETE FROM {} WHERE YEAROOFDATARECORD = '{}'",
                    template.getTableName() + "_TEMP", template.getYear() + "年");
            //sql =  sql + " WHERE YEAR = '"+template.getYear()+"'";
            log.info("执行的sql为:[{}]", sql);
            jdbcTemplate.execute(sql);
        } catch (Exception e) {
            throw new RuntimeException(e);
        } finally {
            DataSourceService.switchDefault();
        }
        return AjaxResult.success();
    }

    @IgnoreWebSecurity
    @PostMapping("/syncDb")
    @ApiOperation("一键导入正式库")
    public AjaxResult syncDb(@RequestBody TemplateVO vo) {
        final DataTemplate template = dataTemplateService.getById(vo.getId());
        if (template == null) {
            return AjaxResult.error("没有对应的模板");
        }
        final long count = excelDataService.count(Wrappers.lambdaQuery(ExcelData.class).eq(ExcelData::getTemplateId, template.getId())
                .eq(ExcelData::getType, "1"));
        if (count < 1) {
            return AjaxResult.error("请先上传导入数据");
        }
        try {
            DataSourceService.switchDb(template.getDataSourceId());
            String insertSql = "INSERT INTO {} SELECT * FROM {} WHERE YEAROOFDATARECORD = '{}年'";
            String deleteSql = "DELETE FROM {} WHERE YEAROOFDATARECORD='{}年'";
            jdbcTemplate.execute(StrFormatter.format(deleteSql, template.getTableName(), template.getYear()));
            jdbcTemplate.execute(StrFormatter.format(insertSql,
                    template.getTableName(), template.getTableName() + "_TEMP", template.getYear()));
        } catch (Exception e) {
            throw new BaseException(e.getMessage());
        } finally {
            DataSourceService.switchDefault();
        }
        DataTemplate updateTemplate = new DataTemplate();
        updateTemplate.setId(template.getId());
        updateTemplate.setImportStatus("1");
        updateTemplate.setImportTime(new Date());
        dataTemplateService.updateById(updateTemplate);
        return AjaxResult.success();
    }





}
