package com.tbyf.his.web.controller.dataImport;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.tbyf.his.common.annotation.IgnoreWebSecurity;
import com.tbyf.his.common.core.domain.AjaxResult;
import com.tbyf.his.common.core.text.StrFormatter;
import com.tbyf.his.common.enums.DataSourceType;
import com.tbyf.his.common.utils.StringUtils;
import com.tbyf.his.common.utils.bean.BeanUtils;
import com.tbyf.his.framework.datasource.DataSourceUtil;
import com.tbyf.his.web.dataImport.domain.param.*;
import com.tbyf.his.web.dataImport.domain.vo.FieldErrorVo;
import com.tbyf.his.web.dataImport.entity.*;
import com.tbyf.his.web.dataImport.mapper.MetaFieldMapper;
import com.tbyf.his.web.dataImport.service.*;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.DataAccessException;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.util.CollectionUtils;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.stream.Collectors;

/**
 * @author lzz
 * @date 2023/2/7 10:42
 */
@RestController
@Api(tags = "数据模板接口")
@RequestMapping("/data/template")
@Slf4j
public class DataTemplateController {

    @Autowired
    private DataTemplateService dataTemplateService;

    @Autowired
    private DataFieldService dataFieldService;

    @Autowired
    private DataRuleService dataRuleService;

    @Autowired
    private BindRuleService bindRuleService;

    @Autowired
    private ExcelDataService excelDataService;

    @Autowired
    private JdbcTemplate jdbcTemplate;

    @Autowired
    private MetaFieldMapper metaFieldMapper;

    @IgnoreWebSecurity
    @GetMapping()
    @ApiOperation("模板查询")
    public AjaxResult queryTemplate(@Validated QueryTemplateParam param) {
        return AjaxResult.success(dataTemplateService.queryTemplate(param));
    }


    @IgnoreWebSecurity
    @GetMapping("/fieldError")
    @ApiOperation("物理表异常字段查询")
    public AjaxResult fieldError(@RequestParam String templateId) throws ExecutionException, InterruptedException {
        List<FieldErrorVo> fieldErrorVos = metaFieldMapper.selectFieldErrorVO(templateId);
        ArrayList<FieldErrorVo> resultList = new ArrayList<>();
        if (!CollectionUtils.isEmpty(fieldErrorVos) && StringUtils.isNotBlank(fieldErrorVos.get(0).getTableName())) {
            CompletableFuture<List<FieldErrorVo>> future1 = dataTemplateService.getTableFieldError(fieldErrorVos, fieldErrorVos.get(0).getTableName());
            CompletableFuture<List<FieldErrorVo>> future2 = dataTemplateService.getTableFieldError(fieldErrorVos, fieldErrorVos.get(0).getTableName() + "_TEMP");
            resultList.addAll(future1.get()); resultList.addAll(future2.get());
        }
        return AjaxResult.success(resultList);
    }

    @IgnoreWebSecurity
    @PostMapping("/syncFieldError")
    @ApiOperation("物理表异常字段同步")
    public AjaxResult fieldErrorUpdate(@RequestBody List<FieldErrorVo> fieldErrorVos) {
        final String alterSql = "ALTER TABLE {} MODIFY ( {} {} )";
        try {
            DataSourceUtil.switchDs(DataSourceType.SLAVE.name());
            fieldErrorVos.forEach(item -> {
                String fieldType = item.getFieldType();
                String fieldName = item.getFieldName();
                String tableName = item.getTableName();
                String sql = StrFormatter.format(alterSql, tableName, fieldName, fieldType);
                try {
                    jdbcTemplate.execute(sql);
                } catch (DataAccessException e) {
                    log.error("alter table's fieldType error");
                    e.printStackTrace();
                }
            });
        } finally {
            DataSourceUtil.switchDefaultDs();
        }
        return AjaxResult.success();
    }

    @IgnoreWebSecurity
    @GetMapping("/allField")
    @ApiOperation("查询模板的所有字段数据")
    public AjaxResult getAllTemplateField(@RequestParam String templateId) {
        return AjaxResult.success(dataFieldService.list(Wrappers.lambdaQuery(DataField.class)
                .eq(DataField::getTemplateId, templateId)));
    }

    @IgnoreWebSecurity
    @GetMapping("/allRule")
    @ApiOperation("查询模板的所有规则数据")
    public AjaxResult getAllTemplateRule(@RequestParam String templateId) {
        final List<BindRule> bindRules = bindRuleService.list(Wrappers.lambdaQuery(BindRule.class)
                .eq(BindRule::getDataId, templateId));
        if (CollectionUtils.isEmpty(bindRules)) {
            return AjaxResult.success(Collections.emptyList());
        }
        return AjaxResult.success(dataRuleService.list(Wrappers.lambdaQuery(DataRule.class)
                .eq(DataRule::getType, "模板规则")
                .in(DataRule::getId, bindRules.stream().map(BindRule::getRuleId).collect(Collectors.toList()))));
    }

    @IgnoreWebSecurity
    @PostMapping("")
    @ApiOperation("模板新增")
    public AjaxResult addTemplate(@RequestBody @Validated AddTemplateParam param) {
        DataTemplate template = new DataTemplate();
        BeanUtils.copyProperties(param, template);
        template.initAdd();
        dataTemplateService.save(template);
        return AjaxResult.success();
    }

    @IgnoreWebSecurity
    @PostMapping("/update")
    @ApiOperation("修改模板")
    public AjaxResult updateTemplate(@RequestBody @Validated UpdateTemplateParam param) {
        DataTemplate template = new DataTemplate();
        BeanUtils.copyProperties(param, template);
        template.initAdd();
        dataTemplateService.updateById(template);
        return AjaxResult.success();
    }

    @IgnoreWebSecurity
    @GetMapping("/delete")
    @ApiOperation("删除模板")
    public AjaxResult deleteTemplate(@RequestParam String templateId) {
        dataTemplateService.removeById(templateId);
        excelDataService.remove(Wrappers.lambdaQuery(ExcelData.class).eq(ExcelData::getTemplateId, templateId));
        final LambdaQueryWrapper<DataField> wrapper = Wrappers.lambdaQuery(DataField.class).eq(DataField::getTemplateId, templateId);
        final List<DataField> list = dataFieldService.list(wrapper);
        final List<String> idList = new ArrayList<>();
        idList.add(templateId);
        if (!CollectionUtils.isEmpty(list)) {
            final List<String> fieldIdList = list.stream().map(DataField::getId).collect(Collectors.toList());
            dataFieldService.remove(Wrappers.lambdaQuery(DataField.class).eq(DataField::getTemplateId, templateId));
            idList.addAll(fieldIdList);
        }
        bindRuleService.remove(Wrappers.lambdaQuery(BindRule.class).in(BindRule::getDataId, idList));

        //TODO 已经删除:模板,字段,模板与字段关联的规则信息,模板文件表  未删除: 模板规则信息
        return AjaxResult.success();
    }

    @IgnoreWebSecurity
    @PostMapping("/rule")
    @ApiOperation("规则新增By Template")
    public AjaxResult addRuleByTemplate(@RequestBody @Validated AddTemplateRuleParam param) {
        DataRule rule = new DataRule();
        BeanUtils.copyProperties(param, rule);
        dataRuleService.save(rule);
        final BindRule bindRule = new BindRule();
        bindRule.setDataId(param.getTid())
                .setRuleId(rule.getId());
        bindRuleService.save(bindRule);
        return AjaxResult.success();
    }

    @IgnoreWebSecurity
    @PostMapping("/rule/update")
    @ApiOperation("修改规则By Template")
    public AjaxResult updateRuleByTemplate(@RequestBody @Validated UpdateRuleParam param) {
        DataRule rule = new DataRule();
        BeanUtils.copyProperties(param, rule);
        dataRuleService.updateById(rule);
        return AjaxResult.success();
    }

    @IgnoreWebSecurity
    @GetMapping("/rule/delete")
    @ApiOperation("删除规则By Template")
    public AjaxResult deleteRuleByTemplate(@RequestParam String ruleId,
                                           @RequestParam String templateId) {
        dataRuleService.removeById(ruleId);
        bindRuleService.remove(Wrappers.lambdaQuery(BindRule.class)
                .eq(BindRule::getRuleId, ruleId)
                .eq(BindRule::getDataId, templateId));
        return AjaxResult.success();
    }

    @IgnoreWebSecurity
    @PostMapping("/otherYears")
    @ApiOperation("查询此类型模板的其它年份列表")
    public AjaxResult queryOtherYears(@RequestBody @Validated DataTemplate template) {
        LambdaQueryWrapper<DataTemplate> queryWrapper = Wrappers.lambdaQuery(DataTemplate.class)
                .eq(DataTemplate::getOrgName, template.getOrgName())
                .ne(DataTemplate::getYear, template.getYear())
                .select(DataTemplate::getYear);
        List<DataTemplate> list = dataTemplateService.list(queryWrapper);
        return AjaxResult.success(list.stream().map(DataTemplate::getYear).filter(StringUtils::isNotBlank).collect(Collectors.toList()));
    }

}
