package com.tbyf.his.system.service.impl;

import com.tbyf.his.common.config.RuoYiConfig;
import com.tbyf.his.common.constant.UserConstants;
import com.tbyf.his.common.core.domain.model.MyKeyValue;
import com.tbyf.his.common.exception.ServiceException;
import com.tbyf.his.common.utils.StringUtils;
import com.tbyf.his.common.utils.bean.BeanValidators;
import com.tbyf.his.common.utils.file.FileUploadUtils;
import com.tbyf.his.framework.config.ServerConfig;
import com.tbyf.his.system.domain.SysPlugin;
import com.tbyf.his.system.mapper.SysPluginMapper;
import com.tbyf.his.system.service.ISysPluginService;
import com.tbyf.his.system.vo.PluginFileVO;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.multipart.MultipartFile;

import javax.validation.Validator;
import java.io.File;
import java.io.IOException;
import java.util.List;
import java.util.Optional;

/**
 * 插件 业务层处理
 */
@Service
public class SysPluginServiceImpl implements ISysPluginService {

    private static final Logger log = LoggerFactory.getLogger(SysPluginServiceImpl.class);

    @Autowired
    private SysPluginMapper pluginMapper;

    @Autowired
    protected Validator validator;

    @Autowired
    private SysPluginAsyncUpload sysPluginAsyncUpload;

    @Autowired
    private ServerConfig serverConfig;

    /**
     * 根据条件分页查询插件列表
     *
     * @param plugin 插件信息
     * @return 插件信息集合信息
     */
    @Override
    public List<SysPlugin> selectPluginList(SysPlugin plugin) {
        return pluginMapper.selectPluginList(plugin);
    }

    /**
     * 通过插件名查询插件
     *
     * @param pluginName 插件名
     * @return 插件对象信息
     */
    @Override
    public SysPlugin selectPluginByPluginName(String pluginName) {
        return pluginMapper.selectPluginByPluginName(pluginName);
    }

    /**
     * 通过插件ID查询插件
     *
     * @param pluginId 插件ID
     * @return 用户对象信息
     */
    @Override
    public SysPlugin selectPluginById(Long pluginId) {
        return pluginMapper.selectPluginById(pluginId);
    }

    /**
     * 校验插件名称是否唯一
     *
     * @param pluginName 插件名称
     * @return 结果
     */
    @Override
    public String checkPluginNameUnique(String pluginName) {
        int count = pluginMapper.checkPluginNameUnique(pluginName);
        if (count > 0) {
            return UserConstants.NOT_UNIQUE;
        }
        return UserConstants.UNIQUE;
    }

    /**
     * 新增保存插件信息
     *
     * @param plugin 插件信息
     * @return 结果
     */
    @Override
    @Transactional
    public int insertPlugin(SysPlugin plugin) {
        // 新增插件信息
        int rows = pluginMapper.insertPlugin(plugin);
        return rows;
    }

    /**
     * 修改保存插件信息
     *
     * @param plugin 插件信息
     * @return 结果
     */
    @Override
    @Transactional
    public int updatePlugin(SysPlugin plugin) {
        Long pluginId = plugin.getPluginId();
        return pluginMapper.updatePlugin(plugin);
    }

    /**
     * 修改插件状态
     *
     * @param plugin 插件信息
     * @return 结果
     */
    @Override
    public int updatePluginStatus(SysPlugin plugin) {
        return pluginMapper.updatePlugin(plugin);
    }

    /**
     * 修改插件基本信息
     *
     * @param plugin 插件信息
     * @return 结果
     */
    @Override
    public int updatePluginProfile(SysPlugin plugin) {
        return pluginMapper.updatePlugin(plugin);
    }

    /**
     * 通过插件ID删除用户
     *
     * @param pluginId 用户ID
     * @return 结果
     */
    @Override
    @Transactional
    public int deletePluginById(Long pluginId) {
        return pluginMapper.deletePluginById(pluginId);
    }

    /**
     * 批量删除插件信息
     *
     * @param pluginIds 需要删除的用户ID
     * @return 结果
     */
    @Override
    @Transactional
    public int deletePluginByIds(Long[] pluginIds) {
        return pluginMapper.deletePluginByIds(pluginIds);
    }

    /**
     * 导入插件数据
     *
     * @param pluginList      插件数据列表
     * @param isUpdateSupport 是否更新支持，如果已存在，则进行更新数据
     * @param operName        操作用户
     * @return 结果
     */
    @Override
    public String importPlugin(List<SysPlugin> pluginList, Boolean isUpdateSupport, String operName) {
        if (StringUtils.isNull(pluginList) || pluginList.size() == 0) {
            throw new ServiceException("导入用户数据不能为空！");
        }
        int successNum = 0;
        int failureNum = 0;
        StringBuilder successMsg = new StringBuilder();
        StringBuilder failureMsg = new StringBuilder();
        for (SysPlugin plugin : pluginList) {
            try {
                // 验证是否存在这个插件
                SysPlugin p = pluginMapper.selectPluginByPluginName(plugin.getPluginName());
                if (StringUtils.isNull(p)) {
                    BeanValidators.validateWithException(validator, plugin);
                    plugin.setCreateBy(operName);
                    this.insertPlugin(plugin);
                    successNum++;
                    successMsg.append("<br/>" + successNum + "、插件 " + plugin.getPluginName() + " 导入成功");
                } else if (isUpdateSupport) {
                    BeanValidators.validateWithException(validator, plugin);
                    plugin.setUpdateBy(operName);
                    this.updatePlugin(plugin);
                    successNum++;
                    successMsg.append("<br/>" + successNum + "、插件 " + plugin.getPluginName() + " 更新成功");
                } else {
                    failureNum++;
                    failureMsg.append("<br/>" + failureNum + "、插件 " + plugin.getPluginName() + " 已存在");
                }
            } catch (Exception e) {
                failureNum++;
                String msg = "<br/>" + failureNum + "、插件 " + plugin.getPluginName() + " 导入失败：";
                failureMsg.append(msg + e.getMessage());
                log.error(msg, e);
            }
        }
        if (failureNum > 0) {
            failureMsg.insert(0, "很抱歉，导入失败！共 " + failureNum + " 条数据格式不正确，错误如下：");
            throw new ServiceException(failureMsg.toString());
        } else {
            successMsg.insert(0, "恭喜您，数据已全部导入成功！共 " + successNum + " 条，数据如下：");
        }
        return successMsg.toString();
    }

    @Override
    public PluginFileVO uploadFile(MultipartFile file) {
        // 用绝对路径，才能用transferTo , win下用 \\ , linux下用 /
//        String absolutePath = new File("").getAbsolutePath() + "\\" + file.getOriginalFilename();
//        System.out.println(absolutePath);
//        sysPluginAsyncUpload.uploadFile(file, absolutePath);
        String url = null;
        try {
            // 上传文件路径yml中配置
            String filePath = RuoYiConfig.getUploadPath();
            // 上传并返回新文件名称
            String fileName = FileUploadUtils.upload(filePath, file);
            url = serverConfig.getUrl() + fileName;
            sysPluginAsyncUpload.uploadFile(file, url);
        } catch (IOException e) {
            e.printStackTrace();
        }

        return PluginFileVO.builder()
                .fileName(file.getOriginalFilename())
                .filePath(url)
                .build();
    }

    @Override
    public File getLocalFile(Long pluginId) {
        SysPlugin sysPlugin = Optional.of(selectPluginById(pluginId))
                .orElseThrow(() -> new ServiceException("未找到对应插件"));
        File file = new File(sysPlugin.getPluginPath());
        if (!file.exists()) {
            throw new ServiceException("本地未找到该文件");
        }
        return file;
    }

    @Override
    public List<MyKeyValue> getOptionByCategory(Long categoryId) {
        return pluginMapper.selectByCategory(categoryId);
    }
}
