package com.tbyf.his.login.service;

import com.tbyf.his.common.constant.Constants;
import com.tbyf.his.common.core.domain.entity.SysUser;
import com.tbyf.his.common.core.domain.model.LoginUser;
import com.tbyf.his.common.core.redis.RedisCache;
import com.tbyf.his.common.enums.UserStatus;
import com.tbyf.his.common.exception.ServiceException;
import com.tbyf.his.common.exception.user.CaptchaException;
import com.tbyf.his.common.exception.user.CaptchaExpireException;
import com.tbyf.his.common.exception.user.UserPasswordNotMatchException;
import com.tbyf.his.common.utils.*;
import com.tbyf.his.common.utils.ip.IpUtils;
import com.tbyf.his.framework.manager.AsyncManager;
import com.tbyf.his.framework.manager.factory.AsyncFactory;
import com.tbyf.his.framework.system.service.ISysConfigService;
import com.tbyf.his.framework.system.service.ISysUserService;
import com.tbyf.his.framework.web.service.SysPermissionService;
import com.tbyf.his.framework.web.service.TokenService;
import com.tbyf.his.system.domain.SysApplication;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.codec.digest.DigestUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cache.CacheManager;
import org.springframework.security.authentication.AuthenticationManager;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.io.UnsupportedEncodingException;
import java.util.List;

/**
 * 登录校验方法
 *
 * @author guopx
 */
@Component
@Slf4j
public class SysLoginService {
    @Autowired
    private TokenService tokenService;

    @Resource
    private AuthenticationManager authenticationManager;

    @Autowired
    private RedisCache redisCache;

    @Autowired
    private ISysUserService userService;

    @Autowired
    private ISysConfigService configService;

    @Autowired
    private SysPermissionService permissionService;
    @Autowired
    private CaptchaService captchaService;

    @Autowired
    private CacheManager cacheManager;

    @Autowired
    private com.tbyf.his.system.mapper.SysApplicationMapper sysApplicationMapper;

    /**
     * 登录验证
     *
     * @param username 用户名
     * @param password 密码
     * @param code     验证码
     * @param uuid     唯一标识
     * @return 结果
     */
    public String login(String username, String password, String code, String uuid) {
        boolean captchaOnOff = configService.selectCaptchaOnOff();
//        boolean captchaOnOff = true;
        // 验证码开关
        if (captchaOnOff) {
            String captcha = captchaService.cacheCaptcha(uuid, code);
            if (captcha == null) {
                AsyncManager.me().execute(AsyncFactory.recordLogininfor(username, Constants.LOGIN_FAIL, MessageUtils.message("user.jcaptcha.expire")));
                throw new CaptchaExpireException();
            }
            if (!code.equalsIgnoreCase(captcha)) {
                AsyncManager.me().execute(AsyncFactory.recordLogininfor(username, Constants.LOGIN_FAIL, MessageUtils.message("user.jcaptcha.error")));
                throw new CaptchaException();
            }
            captchaService.removeCaptcha(uuid);
        }
        // 用户验证
        SysUser user = userService.selectUserByUserName(username);
        checkUser(username, user);
        //获取数据库查询的用户对应的加密密码
        //对用户输入的密码加密
        boolean pass = SecurityUtils.matchesPassword(password, user.getPassword());
        LoginUser loginUser;
        if (pass) {
            loginUser = new LoginUser(user.getUserId(), user.getDeptId(), user, permissionService.getMenuPermission(user));
        } else {
            throw new UserPasswordNotMatchException();
        }

        AsyncManager.me().execute(AsyncFactory.recordLogininfor(username, Constants.LOGIN_SUCCESS, MessageUtils.message("user.login.success")));

        recordLoginInfo(loginUser.getUserId());

        // 生成token
        String token = tokenService.createToken(loginUser);
        return token;
    }

    public String loginByAuthCode(String username, String authCode) {
        //验证授权码
        if (StringUtils.isNotEmpty(authCode)) {
            //授权码加密 匹配应用
            String sha256AuthCode;
            try {
                sha256AuthCode = DigestUtils.sha256Hex(authCode.getBytes(Constants.UTF8));
            } catch (UnsupportedEncodingException e) {
                AsyncManager.me().execute(AsyncFactory.recordLogininfor(username, Constants.LOGIN_FAIL, "授权码解析异常"));
                throw new ServiceException("授权码解析异常");
            }
            SysApplication app = new SysApplication();
//            app.setAuthCode(sha256AuthCode);
            app.setAuthCode(authCode);
            List<SysApplication> list = sysApplicationMapper.selectSysApplicationList(app);
            //授权码匹配成功
            if (StringUtils.isEmpty(list)) {
                AsyncManager.me().execute(AsyncFactory.recordLogininfor(username, Constants.LOGIN_FAIL, MessageUtils.message("auth.code.notfound")));
                throw new ServiceException(MessageUtils.message("auth.code.notfound"));
            }

            // 用户验证
            SysUser user = userService.selectUserByUserName(username);
            checkUser(username, user);
            AsyncManager.me().execute(AsyncFactory.recordLogininfor(username, Constants.LOGIN_SUCCESS, MessageUtils.message("user.login.success")));

            LoginUser loginUser = new LoginUser(user.getUserId(), user.getDeptId(), user, permissionService.getMenuPermission(user));

            recordLoginInfo(loginUser.getUserId());

            // 生成token
            String token = tokenService.createToken(loginUser);
            return token;
        } else {
            AsyncManager.me().execute(AsyncFactory.recordLogininfor(username, Constants.LOGIN_FAIL, MessageUtils.message("auth.code.notfound")));
            throw new ServiceException(MessageUtils.message("auth.code.notfound"));
        }

    }

    private void checkUser(String username, SysUser user) {
        if (StringUtils.isNull(user)) {
            log.info("登录用户：{} 不存在.", username);
            throw new ServiceException("登录用户：" + username + " 不存在");
        } else if (UserStatus.DELETED.getCode().equals(user.getDelFlag())) {
            log.info("登录用户：{} 已被删除.", username);
            throw new ServiceException("对不起，您的账号：" + username + " 已被删除");
        } else if (UserStatus.DISABLE.getCode().equals(user.getStatus())) {
            log.info("登录用户：{} 已被停用.", username);
            throw new ServiceException("对不起，您的账号：" + username + " 已停用");
        }
    }

    /**
     * 校验验证码
     *
     * @param username 用户名
     * @param code     验证码
     * @param uuid     唯一标识
     * @return 结果
     */
    public void validateCaptcha(String username, String code, String uuid) {
        String verifyKey = Constants.CAPTCHA_CODE_KEY + StringUtils.nvl(uuid, "");
        String captcha = redisCache.getCacheObject(verifyKey);
        redisCache.deleteObject(verifyKey);
        if (captcha == null) {
            AsyncManager.me().execute(AsyncFactory.recordLogininfor(username, Constants.LOGIN_FAIL, MessageUtils.message("user.jcaptcha.expire")));
            throw new CaptchaExpireException();
        }
        if (!code.equalsIgnoreCase(captcha)) {
            AsyncManager.me().execute(AsyncFactory.recordLogininfor(username, Constants.LOGIN_FAIL, MessageUtils.message("user.jcaptcha.error")));
            throw new CaptchaException();
        }
    }

    /**
     * 记录登录信息
     *
     * @param userId 用户ID
     */
    public void recordLoginInfo(Long userId) {
        SysUser sysUser = new SysUser();
        sysUser.setUserId(userId);
        sysUser.setLoginIp(IpUtils.getIpAddr(ServletUtils.getRequest()));
        sysUser.setLoginDate(DateUtils.getNowDate());
        userService.updateUserProfile(sysUser);
    }
}
