package com.tbyf.his.web.controller.dataImport;

import com.alibaba.druid.pool.DruidDataSource;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.tbyf.his.common.annotation.IgnoreWebSecurity;
import com.tbyf.his.common.core.domain.AjaxResult;
import com.tbyf.his.common.core.page.TableDataInfo;
import com.tbyf.his.common.core.text.StrFormatter;
import com.tbyf.his.common.utils.StringUtils;
import com.tbyf.his.common.utils.bean.BeanUtils;
import com.tbyf.his.framework.datasource.DataSourceUtil;
import com.tbyf.his.framework.datasource.DynamicDataSourceContextHolder;
import com.tbyf.his.system.domain.SysDatasource;
import com.tbyf.his.system.service.ISysDatasourceService;
import com.tbyf.his.web.dataImport.core.DiConfig;
import com.tbyf.his.web.dataImport.core.RuleVO;
import com.tbyf.his.web.dataImport.core.RuleValidator;
import com.tbyf.his.web.dataImport.domain.param.*;
import com.tbyf.his.web.dataImport.entity.*;
import com.tbyf.his.web.dataImport.service.*;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.ss.usermodel.WorkbookFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;

/**
 * @author lzz
 * @date 2023/2/7 10:42
 */
@RestController
@Api(tags = "数据导入接口")
@RequestMapping("/dataImport")
@Slf4j
public class DataImportController {

    @Autowired
    private DataImportTemplateService dataImportService;

    @Autowired
    private DataDictService dictService;

    @Autowired
    private ISysDatasourceService sysDatasourceService;

    @Autowired
    private JdbcTemplate jdbcTemplate;

    @Autowired
    private DataFieldService dataFieldService;

    @Autowired
    private DataRuleService dataRuleService;

    @Autowired
    private BindRuleService bindRuleService;

    @Autowired
    private ExcelDataService excelDataService;

    @IgnoreWebSecurity
    @GetMapping("/template")
    @ApiOperation("模板查询")
    public AjaxResult queryTemplate(@Validated QueryTemplateParam param) {
        return AjaxResult.success(dataImportService.queryTemplate(param));
    }

    @IgnoreWebSecurity
    @PostMapping("/template")
    @ApiOperation("模板新增")
    public AjaxResult addTemplate(@RequestBody @Validated AddTemplateParam param) {
        DataImportTemplate template = new DataImportTemplate();
        BeanUtils.copyProperties(param, template);
        template.initAdd();
        dataImportService.save(template);
        return AjaxResult.success();
    }

    @IgnoreWebSecurity
    @PutMapping("/template")
    @ApiOperation("修改模板")
    public AjaxResult updateTemplate(@RequestBody @Validated UpdateTemplateParam param) {
        DataImportTemplate template = new DataImportTemplate();
        BeanUtils.copyProperties(param, template);
        template.initAdd();
        dataImportService.updateById(template);
        return AjaxResult.success();
    }

    @IgnoreWebSecurity
    @DeleteMapping("/template")
    @ApiOperation("删除模板")
    public AjaxResult deleteTemplate(@RequestParam String templateId) {
        dataImportService.removeById(templateId);
        excelDataService.remove(Wrappers.lambdaQuery(ExcelData.class).eq(ExcelData::getTemplateId, templateId));
        final LambdaQueryWrapper<DataField> wrapper = Wrappers.lambdaQuery(DataField.class).eq(DataField::getTemplateId, templateId);
        final List<DataField> list = dataFieldService.list(wrapper);
        final List<String> idList = new ArrayList<>();
        idList.add(templateId);
        if (!CollectionUtils.isEmpty(list)) {
            final List<String> fieldIdList = list.stream().map(DataField::getId).collect(Collectors.toList());
            dataFieldService.remove(Wrappers.lambdaQuery(DataField.class).eq(DataField::getTemplateId, templateId));
            idList.addAll(fieldIdList);
        }
        bindRuleService.remove(Wrappers.lambdaQuery(BindRule.class).in(BindRule::getDataId, idList));

        //TODO 已经删除:模板,字段,模板与字段关联的规则信息,模板文件表  未删除: 模板规则信息
        return AjaxResult.success();
    }

    @IgnoreWebSecurity
    @GetMapping("/dict")
    @ApiOperation("字典查询")
    public AjaxResult queryDict(@RequestParam String type) {
        final LambdaQueryWrapper<DataDict> wrapper = Wrappers.lambdaQuery(DataDict.class)
                .eq(DataDict::getType, type);
        final List<Map<String, String>> list = dictService.list(wrapper).stream().map(item -> {
            String json = item.getRemarks();
            item.setRemarks(null);
            final Map<String, String> map = BeanUtils.getFieldValueMap(item);
            if (StringUtils.isNotBlank(json)) {
                try {
                    final JSONObject object = JSON.parseObject(json);
                    object.forEach((key, value) -> {
                        map.put(key, (String) value);
                    });
                } catch (Exception ignore) {
                }
            }
            return map;
        }).collect(Collectors.toList());
        return AjaxResult.success(list);
    }

    @IgnoreWebSecurity
    @PostMapping("/dict")
    @ApiOperation("新增字典")
    public AjaxResult addDict(@RequestBody AddDictParam param) {
        DataDict dict = new DataDict();
        BeanUtils.copyProperties(param, dict);
        dictService.save(dict);
        return AjaxResult.success();
    }

    @IgnoreWebSecurity
    @PutMapping("/dict")
    @ApiOperation("修改字典")
    public AjaxResult updateDict(@RequestBody @Validated UpdateDictParam param) {
        DataDict template = new DataDict();
        BeanUtils.copyProperties(param, template);
        dictService.updateById(template);
        return AjaxResult.success();
    }

    @IgnoreWebSecurity
    @DeleteMapping("/dict")
    @ApiOperation("删除字典")
    public AjaxResult deleteDict(@RequestParam String dictId) {
        dictService.removeById(dictId);
        return AjaxResult.success();
    }

    @IgnoreWebSecurity
    @GetMapping("/datasource")
    @ApiOperation("查询所有数据源")
    public AjaxResult queryDatasource() {
        final List<SysDatasource> list = sysDatasourceService.selectSysDatasourceList(new SysDatasource());
        return AjaxResult.success(list);
    }

    @IgnoreWebSecurity
    @GetMapping("/datasource/table")
    @ApiOperation("获取指定数据源中的表")
    public AjaxResult getDbTable(@RequestParam String dataSourceId) {
        try {
            DataSourceUtil.switchDs(dataSourceId);
            final DruidDataSource dataSource = (DruidDataSource) DynamicDataSourceContextHolder.dataSourcesMap.get(dataSourceId);
            String sql = "select  TABLE_NAME AS VALUE,COMMENTS AS LABEL from user_tab_comments WHERE TABLE_TYPE = 'TABLE'";
            if (dataSource.getDriverClassName().toLowerCase().contains("mysql")) {
                sql = "select  TABLE_NAME AS VALUE ,table_comment AS LABEL from information_schema.tables where  table_type='BASE TABLE'";
            }
            final List<Map<String, Object>> maps = jdbcTemplate.queryForList(sql);
            return AjaxResult.success(maps.stream().filter(
                    distinctByKey(b -> b.get("VALUE"))
            ).distinct().collect(Collectors.toList()));
        } finally {
            DataSourceUtil.switchDefaultDs();
        }
    }

    @IgnoreWebSecurity
    @GetMapping("/datasource/field")
    @ApiOperation("获取指定数据表中的字段")
    public AjaxResult getDbField(@RequestParam String dataSourceId, @RequestParam String tableName) {
        try {
            DataSourceUtil.switchDs(dataSourceId);
            final DruidDataSource dataSource = (DruidDataSource) DynamicDataSourceContextHolder.dataSourcesMap.get(dataSourceId);
            String sql = StrFormatter.format("SELECT UTC.COLUMN_NAME AS VALUE,UTC.DATA_TYPE AS TYPE,( SELECT  UCC.COMMENTS  FROM user_col_comments UCC WHERE UCC.COLUMN_NAME = UTC.COLUMN_NAME AND UCC.TABLE_NAME = UTC.TABLE_NAME ) AS LABEL FROM user_tab_columns UTC WHERE UTC.TABLE_NAME = '{}'", tableName);
            if (dataSource.getDriverClassName().toLowerCase().contains("mysql")) {
                sql = StrFormatter.format("SELECT column_name AS VALUE, DATA_TYPE AS TYPE, column_comment AS LABEL FROM information_schema.COLUMNS WHERE table_name = '{}'", tableName);
            }
            final List<Map<String, Object>> maps = jdbcTemplate.queryForList(sql);
            return AjaxResult.success(maps);
        } finally {
            DataSourceUtil.switchDefaultDs();
        }
    }


    public static <T> Predicate<T> distinctByKey(Function<? super T, ?> keyExtractor) {
        Map<Object, Boolean> seen = new ConcurrentHashMap<>();
        return t -> seen.putIfAbsent(keyExtractor.apply(t), Boolean.TRUE) == null;
    }


    @IgnoreWebSecurity
    @GetMapping("/field")
    @ApiOperation("字段查询")
    public TableDataInfo queryField(@Validated QueryFieldParam param) {
        final Page<DataField> page = Page.of(param.getPageNum(), param.getPageSize());
        final LambdaQueryWrapper<DataField> queryWrapper = Wrappers.lambdaQuery(DataField.class);
        queryWrapper.eq(StringUtils.isNotBlank(param.getTemplateId()), DataField::getTemplateId, param.getTemplateId())
                .like(StringUtils.isNotBlank(param.getCode()), DataField::getCode, param.getCode())
                .like(StringUtils.isNotBlank(param.getTitle()), DataField::getTitle, param.getTitle())
                .like(StringUtils.isNotBlank(param.getCoordinate()), DataField::getCoordinate, param.getCoordinate())
                .like(StringUtils.isNotBlank(param.getField()), DataField::getField, param.getField());
        final Page<DataField> templatePage = dataFieldService.page(page, queryWrapper);
        return param.convert(templatePage);
    }

    @IgnoreWebSecurity
    @PostMapping("/field")
    @ApiOperation("字段新增")
    public AjaxResult addField(@RequestBody @Validated AddFieldParam param) {
        DataField field = new DataField();
        BeanUtils.copyProperties(param, field);
        dataFieldService.save(field);
        return AjaxResult.success();
    }

    @IgnoreWebSecurity
    @PutMapping("/field")
    @ApiOperation("修改字段")
    public AjaxResult updateField(@RequestBody @Validated UpdateFieldParam param) {
        DataField field = new DataField();
        BeanUtils.copyProperties(param, field);
        dataFieldService.updateById(field);
        return AjaxResult.success();
    }

    @IgnoreWebSecurity
    @DeleteMapping("/field")
    @ApiOperation("删除字段")
    public AjaxResult deleteField(@RequestParam String fieldId) {
        dataFieldService.removeById(fieldId);
        bindRuleService.remove(Wrappers.lambdaQuery(BindRule.class).eq(BindRule::getDataId, fieldId));
        return AjaxResult.success();
    }

    @IgnoreWebSecurity
    @GetMapping("/rule")
    @ApiOperation("规则查询")
    public TableDataInfo queryRule(@Validated QueryRuleParam param) {
        final Page<DataRule> page = Page.of(param.getPageNum(), param.getPageSize());
        final LambdaQueryWrapper<DataRule> queryWrapper = Wrappers.lambdaQuery(DataRule.class);
        queryWrapper.like(StringUtils.isNotBlank(param.getName()), DataRule::getName, param.getName())
                .eq(StringUtils.isNotBlank(param.getType()), DataRule::getType, param.getType())
                .in(DataRule::getType, "基础规则", "组合规则");
        final Page<DataRule> templatePage = dataRuleService.page(page, queryWrapper);
        return param.convert(templatePage);
    }

    @IgnoreWebSecurity
    @GetMapping("/rule/dict")
    @ApiOperation("规则字典")
    public AjaxResult queryRuleDict() {
        final LambdaQueryWrapper<DataRule> queryWrapper = Wrappers.lambdaQuery(DataRule.class);
        queryWrapper.in(DataRule::getType, "基础规则", "组合规则")
                .select(DataRule::getId, DataRule::getName);
        return AjaxResult.success(dataRuleService.list(queryWrapper));
    }

    @IgnoreWebSecurity
    @PostMapping("/rule")
    @ApiOperation("规则新增")
    public AjaxResult addRule(@RequestBody @Validated AddRuleParam param) {
        DataRule rule = new DataRule();
        BeanUtils.copyProperties(param, rule);
        dataRuleService.save(rule);
        return AjaxResult.success();
    }

    @IgnoreWebSecurity
    @PutMapping("/rule")
    @ApiOperation("修改规则")
    public AjaxResult updateRule(@RequestBody @Validated UpdateRuleParam param) {
        DataRule rule = new DataRule();
        BeanUtils.copyProperties(param, rule);
        dataRuleService.updateById(rule);
        return AjaxResult.success();
    }

    @IgnoreWebSecurity
    @DeleteMapping("/rule")
    @ApiOperation("删除规则")
    public AjaxResult deleteRule(@RequestParam String ruleId) {
        dataRuleService.removeById(ruleId);
        bindRuleService.remove(Wrappers.lambdaQuery(BindRule.class).eq(BindRule::getRuleId, ruleId));
        return AjaxResult.success();
    }

    @IgnoreWebSecurity
    @PostMapping("/rule/test")
    @ApiOperation("规则测试")
    public AjaxResult testRule(@RequestBody @Validated RuleVO test) {
        final RuleValidator validator = DiConfig.getValidator(test.getMode());
        validator.validate(test);
        if (StringUtils.isNotBlank(test.getResult())) {
            return AjaxResult.error(test.getResult());
        }
        return AjaxResult.success();
    }

    @IgnoreWebSecurity
    @GetMapping("/bind/quick")
    @ApiOperation("快速绑定")
    public AjaxResult quickBind(@RequestParam String templateId,
                                @RequestParam String ruleId) {
        final LambdaQueryWrapper<DataField> queryWrapper = Wrappers.lambdaQuery(DataField.class)
                .select(DataField::getId)
                .eq(DataField::getTemplateId, templateId);
        final List<String> fieldIdList = dataFieldService.list(queryWrapper).stream()
                .map(DataField::getId)
                .collect(Collectors.toList());
        final LambdaQueryWrapper<BindRule> removeWrapper = Wrappers.lambdaQuery(BindRule.class)
                .in(BindRule::getDataId, fieldIdList)
                .eq(BindRule::getRuleId, ruleId);
        bindRuleService.remove(removeWrapper);
        final List<BindRule> bindRuleList = fieldIdList.stream().map(item -> {
            BindRule bind = new BindRule();
            bind.setDataId(item);
            bind.setRuleId(ruleId);
            return bind;
        }).collect(Collectors.toList());
        bindRuleService.saveBatch(bindRuleList);
        return AjaxResult.success();
    }

    @IgnoreWebSecurity
    @GetMapping("/bind")
    @ApiOperation("查询绑定的规则")
    public AjaxResult queryBindRule(@RequestParam String fieldId) {
        final List<BindRule> list = bindRuleService.list(Wrappers.lambdaQuery(BindRule.class)
                .eq(BindRule::getDataId, fieldId));
        if (CollectionUtils.isEmpty(list)) {
            return AjaxResult.success(Collections.emptyList());
        }
        final List<String> idList = list.stream().map(BindRule::getRuleId).collect(Collectors.toList());
        return AjaxResult.success(dataRuleService.list(Wrappers.lambdaQuery(DataRule.class)
                .in(DataRule::getId, idList)));
    }

    @IgnoreWebSecurity
    @PostMapping("/bind")
    @ApiOperation("配置字段规则绑定")
    @Transactional(rollbackFor = Exception.class)
    public AjaxResult bindRule(@RequestBody BindRuleParam param) {
        bindRuleService.remove(Wrappers.lambdaQuery(BindRule.class).eq(BindRule::getDataId, param.getFieldId()));
        if (!CollectionUtils.isEmpty(param.getRuleIdList())) {
            final List<BindRule> bindRuleList = param.getRuleIdList().stream().map(item -> {
                BindRule bd = new BindRule();
                bd.setRuleId(item);
                bd.setDataId(param.getFieldId());
                return bd;
            }).collect(Collectors.toList());
            bindRuleService.saveBatch(bindRuleList);
        }
        return AjaxResult.success();
    }

    @SneakyThrows
    @IgnoreWebSecurity
    @PostMapping("/upload")
    @ApiOperation("文件导入")
    public AjaxResult bindRule(UploadExcelParam param) {
        if (StringUtils.equals(param.getType(), "1")) {

        } else if (StringUtils.equals(param.getType(), "2")) {
            ExcelData excelData = excelDataService.getOne(Wrappers.lambdaQuery(ExcelData.class)
                    .eq(ExcelData::getTemplateId, param.getTemplateId())
                    .eq(ExcelData::getType, "2"), false);
            if (ObjectUtils.isEmpty(excelData)) {
                excelData = new ExcelData();
                excelData.setFile(param.getFile().getBytes());
                excelData.setFileName(param.getFile().getOriginalFilename());
                excelData.setType("2");
                excelData.setTemplateId(param.getTemplateId());
                excelData.setYear(param.getYear());
                excelData.setOrgName(param.getOrgName());
                excelDataService.save(excelData);
            } else {
                excelData.setFile(param.getFile().getBytes());
                excelData.setFileName(param.getFile().getOriginalFilename());
                excelDataService.updateById(excelData);
            }
        }
        return AjaxResult.success();
    }

    @IgnoreWebSecurity
    @GetMapping("/field/reset")
    @ApiOperation("根据基础模板重置字段")
    public AjaxResult resetField(@RequestParam String excelId) {
        final ExcelData excelData = excelDataService.getById(excelId);
        final LambdaQueryWrapper<DataField> wrapper = Wrappers.lambdaQuery(DataField.class).eq(DataField::getTemplateId, excelData.getTemplateId());
        final List<DataField> list = dataFieldService.list(wrapper);
        if (!CollectionUtils.isEmpty(list)) {
            final List<String> fieldIdList = list.stream().map(DataField::getId).collect(Collectors.toList());
            bindRuleService.remove(Wrappers.lambdaQuery(BindRule.class).in(BindRule::getDataId, fieldIdList));
        }
        dataFieldService.remove(Wrappers.lambdaQuery(DataField.class).eq(DataField::getTemplateId, excelData.getTemplateId()));
        try (InputStream is = new ByteArrayInputStream(excelData.getFile()); Workbook workbook = WorkbookFactory.create(is)) {
            final Sheet sheet = workbook.getSheetAt(0);
            final int rows = sheet.getLastRowNum() + 1;
            List<DataField> fieldList = new ArrayList<>();
            for (int i = 1; i < rows; i++) {
                final Row row = sheet.getRow(i);
                String code = DiConfig.getValue(row.getCell(1));
                String title = DiConfig.getValue(row.getCell(2));
                String unit = DiConfig.getValue(row.getCell(4));
                if (StringUtils.isAllBlank(code, title, unit)) {
                    continue;
                }
                if (StringUtils.equals(code, "代码") || StringUtils.equals(title, "指标名称") || StringUtils.equals(unit, "计量单位")) {
                    continue;
                }
                final DataField dataField = new DataField();
                dataField.setTemplateId(excelData.getTemplateId());
                dataField.setCode(code);
                dataField.setTitle(title);
                dataField.setUnit(unit);
                dataField.setCoordinate("F," + (i + 1));
                dataField.setSort(i + 1);
                dataField.createField();
                fieldList.add(dataField);
            }
            dataFieldService.saveBatch(fieldList);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
        return AjaxResult.success();
    }


}
