package com.tbyf.his.apiconvert.service.impl;

import com.tbyf.his.apiconvert.domain.FieldMap;
import com.tbyf.his.apiconvert.domain.model.SourceAndTarget;
import com.tbyf.his.common.core.domain.entity.FieldMapping;
import com.tbyf.his.apiconvert.mapper.FieldMappingMapper;
import com.tbyf.his.apiconvert.service.IFieldMappingService;
import com.tbyf.his.common.constant.UserConstants;
import com.tbyf.his.common.core.domain.TreeSelect;
import com.tbyf.his.common.utils.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 字段管理 服务实现
 *
 * @author guopx
 */
@Service
public class FieldMappingServiceImpl implements IFieldMappingService {
    @Autowired
    private FieldMappingMapper FieldMapper;

    /**
     * 查询字段管理数据
     *
     * @param field 字段信息
     * @return 字段信息集合
     */
    @Override
    public List<FieldMapping> selectFieldList(FieldMapping field) {
        return FieldMapper.selectFieldList(field);
    }

    @Override
    public List<FieldMapping> selectConnectTo(FieldMapping field) {
        return FieldMapper.selectConnectTo(field);
    }

    /**
     * 构建前端所需要树结构
     *
     * @param fields 字段列表
     * @return 树结构列表
     */
    @Override
    public List<FieldMapping> buildFieldTree(List<FieldMapping> fields) {
        List<FieldMapping> returnList = new ArrayList<FieldMapping>();
        List<Long> tempList = new ArrayList<Long>();
        for (FieldMapping Field : fields) {
            tempList.add(Field.getFieldId());
        }
        for (FieldMapping Field : fields) {
            // 如果是顶级节点, 遍历该父节点的所有子节点
            if (!tempList.contains(Field.getParentId())) {
                recursionFn(fields, Field);
                returnList.add(Field);
            }
        }
        if (returnList.isEmpty()) {
            returnList = fields;
        }
        return returnList;
    }

    /**
     * 构建前端所需要下拉树结构
     *
     * @param fields 字段列表
     * @return 下拉树结构列表
     */
    @Override
    public List<TreeSelect> buildFieldTreeSelect(List<FieldMapping> fields) {
        List<FieldMapping> FieldTrees = buildFieldTree(fields);
        return FieldTrees.stream().map(TreeSelect::new).collect(Collectors.toList());
    }

    /**
     * 根据字段ID查询信息
     *
     * @param fieldId 字段ID
     * @return 字段信息
     */
    @Override
    public FieldMapping selectFieldById(Long fieldId) {
        return FieldMapper.selectFieldById(fieldId);
    }

    /**
     * 根据ID查询所有子字段（正常状态）
     *
     * @param fieldId 字段ID
     * @return 子字段数
     */
    @Override
    public int selectNormalChildrenFieldById(Long fieldId) {
        return FieldMapper.selectNormalChildrenFieldById(fieldId);
    }

    /**
     * 是否存在子节点
     *
     * @param fieldId 字段ID
     * @return 结果
     */
    @Override
    public boolean hasChildByFieldId(Long fieldId) {
        int result = FieldMapper.hasChildByFieldId(fieldId);
        return result > 0;
    }

    /**
     * 校验字段名称是否唯一
     *
     * @param field 字段信息
     * @return 结果
     */
    @Override
    public String checkFieldNameUnique(FieldMapping field) {
        Long fieldId = StringUtils.isNull(field.getFieldId()) ? -1L : field.getFieldId();
        FieldMapping info = FieldMapper.checkFieldNameUnique(
                field.getApiId(),
                field.getFieldName(),
                field.getParentId(),
                field.getParamFlag(),
                field.getFieldFlag()
        );
        if (StringUtils.isNotNull(info) && info.getFieldId().longValue() != fieldId.longValue()) {
            return UserConstants.NOT_UNIQUE;
        }
        return UserConstants.UNIQUE;
    }

    /**
     * 新增保存字段信息
     *
     * @param field 字段信息
     * @return 结果
     */
    @Override
    public int insertField(FieldMapping field) {
        FieldMapping info = FieldMapper.selectFieldById(field.getParentId());
        String ancestors = info == null ? "" : info.getAncestors() + "," + field.getParentId();
        field.setAncestors(ancestors);
        return FieldMapper.insertField(field);
    }

    /**
     * 修改保存字段信息
     *
     * @param field 字段信息
     * @return 结果
     */
    @Override
    public int updateField(FieldMapping field) {
        FieldMapping newParentField = FieldMapper.selectFieldById(field.getParentId());
        FieldMapping oldField = FieldMapper.selectFieldById(field.getFieldId());
        if (StringUtils.isNotNull(newParentField) && StringUtils.isNotNull(oldField)) {
            String newAncestors = newParentField.getAncestors() + "," + newParentField.getFieldId();
            String oldAncestors = oldField.getAncestors();
            field.setAncestors(newAncestors);
            updateFieldChildren(field.getFieldId(), newAncestors, oldAncestors);
        }
        int result = FieldMapper.updateField(field);
        return result;
    }

    /**
     * 修改子元素关系
     *
     * @param fieldId      被修改的字段ID
     * @param newAncestors 新的父ID集合
     * @param oldAncestors 旧的父ID集合
     */
    public void updateFieldChildren(Long fieldId, String newAncestors, String oldAncestors) {
        List<FieldMapping> children = FieldMapper.selectChildrenFieldById(fieldId);
        for (FieldMapping child : children) {
            child.setAncestors(child.getAncestors().replaceFirst(oldAncestors, newAncestors));
        }
        if (children.size() > 0) {
            FieldMapper.updateFieldChildren(children);
        }
    }

    /**
     * 删除字段管理信息
     *
     * @param fieldId 字段ID
     * @return 结果
     */
    @Override
    public int deleteFieldById(Long fieldId) {
        return FieldMapper.deleteFieldById(fieldId);
    }

    @Override
    public int insertFieldMap(FieldMap field) {
        return FieldMapper.insertFieldMap(field);
    }

    @Override
    public int deleteFieldMap(Long sourceFieldId, Long targetFieldId, Long apiBaseinfoId) {
        return FieldMapper.deleteFieldMap(sourceFieldId, targetFieldId, apiBaseinfoId);
    }

    @Override
    public List<FieldMap> selectFieldMappingByApiId(Long apiId) {
        return FieldMapper.selectFieldMappingByApiId(apiId);
    }

    /**
     * 递归列表
     */
    private void recursionFn(List<FieldMapping> list, FieldMapping t) {
        // 得到子节点列表
        List<FieldMapping> childList = getChildList(list, t);
        t.setChildren(childList);
        for (FieldMapping tChild : childList) {
            if (hasChild(list, tChild)) {
                recursionFn(list, tChild);
            }
        }
    }

    /**
     * 得到子节点列表
     */
    private List<FieldMapping> getChildList(List<FieldMapping> list, FieldMapping t) {
        List<FieldMapping> tlist = new ArrayList<FieldMapping>();
        Iterator<FieldMapping> it = list.iterator();
        while (it.hasNext()) {
            FieldMapping n = (FieldMapping) it.next();
            if (StringUtils.isNotNull(n.getParentId()) && n.getParentId().longValue() == t.getFieldId().longValue()) {
                tlist.add(n);
            }
        }
        return tlist;
    }

    /**
     * 判断是否有子节点
     */
    private boolean hasChild(List<FieldMapping> list, FieldMapping t) {
        return getChildList(list, t).size() > 0;
    }
}
