package com.tbyf.his.system.service.impl;

import com.alibaba.druid.pool.DruidDataSource;
import com.alibaba.fastjson.JSON;
import com.tbyf.his.apiconvert.domain.ExecutSqlsInfo;
import com.tbyf.his.common.core.domain.model.MyKeyValue;
import com.tbyf.his.common.core.page.TableDataInfo;
import com.tbyf.his.common.core.text.CharsetKit;
import com.tbyf.his.common.exception.ServiceException;
import com.tbyf.his.common.utils.DateUtils;
import com.tbyf.his.common.utils.StringUtils;
import com.tbyf.his.framework.datasource.DataSourceUtil;
import com.tbyf.his.framework.datasource.DynamicDataSourceContextHolder;
import com.tbyf.his.framework.system.util.DictUtils;
import com.tbyf.his.framework.utils.AesUtils;
import com.tbyf.his.system.domain.SysDataTableVo;
import com.tbyf.his.system.domain.SysDatasource;
import com.tbyf.his.system.mapper.SysDatasourceMapper;
import com.tbyf.his.system.service.ISysDatasourceService;
import com.tbyf.his.system.vo.DynamicSqlData;
import com.tbyf.his.system.vo.SqlHandler;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.nio.charset.Charset;
import java.sql.SQLException;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 数据源配置Service业务层处理
 *
 * @author guopx
 * @date 2022-03-01
 */
@Service
public class SysDatasourceServiceImpl implements ISysDatasourceService {

    @Resource
    private SysDatasourceMapper sysDatasourceMapper;

    @Autowired
    private SysDatasourceServiceTransactional sysDatasourceServiceTransactional;

    private static final Logger log = LoggerFactory.getLogger(SysDatasourceServiceImpl.class);


    /**
     * 查询数据源配置
     *
     * @param datasourceId 数据源配置主键
     * @return 数据源配置
     */
    @Override
    public SysDatasource selectSysDatasourceByDatasourceId(Long datasourceId) {
        return sysDatasourceMapper.selectSysDatasourceByDatasourceId(datasourceId);
    }

    /**
     * 查询数据源配置列表
     *
     * @param sysDatasource 数据源配置
     * @return 数据源配置
     */
    @Override
    public List<SysDatasource> selectSysDatasourceList(SysDatasource sysDatasource) {
        return sysDatasourceMapper.selectSysDatasourceList(sysDatasource);
    }

    /**
     * 新增数据源配置
     *
     * @param sysDatasource 数据源配置
     * @return 结果
     */
    @Override
    public int insertSysDatasource(SysDatasource sysDatasource) {
        sysDatasource.setCreateTime(DateUtils.getNowDate());
        //数据源密码加密
        sysDatasource.setPassword(AesUtils.encrypt(sysDatasource.getPassword()));
        return sysDatasourceMapper.insertSysDatasource(sysDatasource);
    }

    /**
     * 修改数据源配置
     *
     * @param sysDatasource 数据源配置
     * @return 结果
     */
    @Override
    public int updateSysDatasource(SysDatasource sysDatasource) {
        sysDatasource.setUpdateTime(DateUtils.getNowDate());

        //更新数据源
        SysDatasource ds = selectSysDatasourceByDatasourceId(sysDatasource.getDatasourceId());
        if (StringUtils.isNotNull(ds)) {

            //密码修改了, 则加密存储
            String password = sysDatasource.getPassword();
            if (!ds.getPassword().equals(password)) {
                sysDatasource.setPassword(AesUtils.encrypt(password));
            }

            if (DynamicDataSourceContextHolder.dataSourcesMap.containsKey(ds.getDatasourceName())) {
                DynamicDataSourceContextHolder.dataSourcesMap.remove(ds.getDatasourceName());
            }
        }
        return sysDatasourceMapper.updateSysDatasource(sysDatasource);
    }

    /**
     * 批量删除数据源配置
     *
     * @param datasourceIds 需要删除的数据源配置主键
     * @return 结果
     */
    @Override
    public int deleteSysDatasourceByDatasourceIds(Long[] datasourceIds) {
        for (Long datasourceId : datasourceIds) {
            //更新数据源
            SysDatasource ds = selectSysDatasourceByDatasourceId(datasourceId);
            if (StringUtils.isNotNull(ds)) {
                if (DynamicDataSourceContextHolder.dataSourcesMap.containsKey(ds.getDatasourceName())) {
                    DynamicDataSourceContextHolder.dataSourcesMap.remove(ds.getDatasourceName());
                }
            }
        }
        return sysDatasourceMapper.deleteSysDatasourceByDatasourceIds(datasourceIds);
    }

    /**
     * 删除数据源配置信息
     *
     * @param datasourceId 数据源配置主键
     * @return 结果
     */
    @Override
    public int deleteSysDatasourceByDatasourceId(Long datasourceId) {
        //更新数据源
        SysDatasource ds = selectSysDatasourceByDatasourceId(datasourceId);
        if (StringUtils.isNotNull(ds)) {
            if (DynamicDataSourceContextHolder.dataSourcesMap.containsKey(ds.getDatasourceName())) {
                DynamicDataSourceContextHolder.dataSourcesMap.remove(ds.getDatasourceName());
            }
        }
        return sysDatasourceMapper.deleteSysDatasourceByDatasourceId(datasourceId);
    }

    @Override
    public List<MyKeyValue> selectSysDatasource() {
        return sysDatasourceMapper.selectSysDatasource();
    }

    @Override
    //@DataSource(value = DataSourceType.SLAVE)
    public List<Map<String, Object>> switchDsAndExecuteSql(Long datasourceId, String sql) {
        switchDsByDsId(datasourceId);
        List<Map<String, Object>> list = sysDatasourceMapper.executeSql(sql);
        DataSourceUtil.switchDefaultDs();
        return list;
    }

    @Override
    public List<ExecutSqlsInfo> switchDsAndExecuteSqls(Long datasourceId, List<ExecutSqlsInfo> sqls) {
        switchDsByDsId(datasourceId);

        List<ExecutSqlsInfo> result = new ArrayList<>();
        for (ExecutSqlsInfo sql : sqls) {
            try {
                List<Map<String, Object>> list = sysDatasourceMapper.executeSql(sql.getSql());
                sql.setResult(list);
                result.add(sql);
            } catch (Exception e) {
                System.out.println(e);
            }

        }
        return result;
    }

    @Override
    public ExecutSqlsInfo switchDsAndExecuteImportSql(Long datasourceId, String sql) {
        ExecutSqlsInfo info = new ExecutSqlsInfo();
        info.setSql(sql);
        switchDsByDsId(datasourceId);
        if (isSelect(sql)) {
            info.setResult(sysDatasourceMapper.executeSql(sql));
        } else {
            sysDatasourceMapper.executeDDL(sql);
        }
        DataSourceUtil.switchDefaultDs();
        info.setSuccess(true);
        return info;
    }

    @Override
    public Map<String, Object> getTablesBySourceId(Long id) {
        SysDatasource datasource = selectSysDatasourceByDatasourceId(id);
        boolean isOracle = SqlHandler.isOracle(datasource.getDriverClass());
        String db = SqlHandler.getDB(isOracle, datasource.getUrl(), datasource.getUsername());
        String sql = SqlHandler.getQueryAllTableSql(isOracle, db);
        log.info("[getTablesBySourceId]: {} - {}", id, sql);
        List<Map<String, Object>> list = switchDsAndExecuteImportSql(id, sql).getResult();
        return new HashMap<String, Object>() {{
            put("isOracle", isOracle);
            put("option", list.stream().map(d -> {
                MyKeyValue item = new MyKeyValue();
                String v = String.valueOf(d.get("TABLE_NAME"));
                String str = String.valueOf(d.get(isOracle ? "COMMENTS" : "TABLE_COMMENT")).trim();
                String k = StringUtils.equalsAny(str, "", "null") ? v : str;
                item.setKey(k);
                item.setValue(v);
                return item;
            }).collect(Collectors.toList()));
        }};
    }

    @Override
    public void testConnectBySourceId(SysDatasource sysDatasource) {
        try {
            DruidDataSource connection = DataSourceUtil.getConnection(sysDatasource.getUrl(), sysDatasource.getDriverClass(), sysDatasource.getUsername(), sysDatasource.getPassword());
            if (connection != null) {
                connection.close();
            } else {
                throw new RuntimeException("连接失败");
            }
        } catch (SQLException e) {
            throw new RuntimeException("连接失败");
        }
    }

    @Override
    public Map<String, Object> getTablesBySourceIdAndTableName(SysDataTableVo sysDataTableVo) {
        SysDatasource datasource = selectSysDatasourceByDatasourceId(sysDataTableVo.getId());
        boolean isOracle = SqlHandler.isOracle(datasource.getDriverClass());
        String db = SqlHandler.getDB(isOracle, datasource.getUrl(), datasource.getUsername());

        String filter = null;
        if (sysDataTableVo.getFilterTableString() != null && sysDataTableVo.getFilterTableString().length > 0) {
            String[] filterString = new String[sysDataTableVo.getFilterTableString().length];
            for (int i = 0; i < sysDataTableVo.getFilterTableString().length; i++) {
                filterString[i] = "'" + sysDataTableVo.getFilterTableString()[i] + "'";
            }
            filter = Arrays.toString(filterString).replace("[", "").replace("]", "");
        } else {
            filter = "'1'";
        }
        String sql = null;
        String selectCountSql = null;
        if (StringUtils.isEmpty(sysDataTableVo.getSourceTableName())) {
            sql = SqlHandler.getQueryAllTablePageSql(isOracle, db, filter, sysDataTableVo.getPageNum(), sysDataTableVo.getPageSize());

            log.info(selectCountSql);
        } else {
            sql = SqlHandler.getQueryAllTablePageAndNameSql(isOracle, db, filter, sysDataTableVo.getSourceTableName(), sysDataTableVo.getPageNum(), sysDataTableVo.getPageSize());
        }
        if (isOracle) {
            selectCountSql = "select count(*) from (" + sql.substring(0, sql.lastIndexOf("WHERE")) + ") temp";
        } else {
            selectCountSql = "select count(*) from (" + sql.substring(0, sql.lastIndexOf("limit")) + ") temp";
        }
        log.info("[getTablesBySourceId]: {} - {}", sysDataTableVo.getId(), sql);
        List<Map<String, Object>> list = switchDsAndExecuteImportSql(sysDataTableVo.getId(), sql).getResult();
        List<Map<String, Object>> result = switchDsAndExecuteImportSql(sysDataTableVo.getId(), selectCountSql).getResult();
        return new HashMap<String, Object>() {{
            put("isOracle", isOracle);
            put("option", list.stream().map(d -> {
                MyKeyValue item = new MyKeyValue();
                String v = String.valueOf(d.get("TABLE_NAME"));
                String str = String.valueOf(d.get(isOracle ? "COMMENTS" : "TABLE_COMMENT")).trim();
                String k = StringUtils.equalsAny(str, "", "null") ? v : str;
                k = k.contains("??") ? v : k;
                item.setKey(k);
                item.setValue(v);
                return item;
            }).collect(Collectors.toList()));
            int total = isOracle ? Integer.parseInt(result.get(0).get("COUNT(*)").toString()) : Integer.parseInt(result.get(0).get("count(*)").toString());
            put("total", total);
        }};
    }

    @Override
    public List<MyKeyValue> getColumnsByTableName(Long id, String tableName) {
        SysDatasource datasource = selectSysDatasourceByDatasourceId(id);
        boolean isOracle = SqlHandler.isOracle(datasource.getDriverClass());
        String db = SqlHandler.getDB(isOracle, datasource.getUrl(), datasource.getUsername());
        String sql = SqlHandler.getQueryAllColumnSql(isOracle, db, tableName);
        log.info("[getColumnsByTableName]: {} - {}", id, sql);
        List<Map<String, Object>> list = switchDsAndExecuteImportSql(id, sql).getResult();
        return list.stream().map(d -> {
            MyKeyValue item = new MyKeyValue();
            String v = String.valueOf(d.get("COLUMN_NAME"));
            String str = String.valueOf(d.get(isOracle ? "COMMENTS" : "COLUMN_COMMENT")).trim();
            String k = StringUtils.equalsAny(str, "", "null") ? v : str;
            k = k.contains("??") ? v : k;
            item.setKey(k);
            item.setValue(v);
            return item;
        }).collect(Collectors.toList());
    }

    @Override
    public TableDataInfo selectPageByParam(DynamicSqlData dynamicSqlData) {
        SysDatasource ds = selectSysDatasourceByDatasourceId(dynamicSqlData.getDatasourceId());
        boolean isOracle = SqlHandler.isOracle(ds.getDriverClass());
        String pageSelectSql = SqlHandler.createPageSelectSql(isOracle, dynamicSqlData.getQueryParam(),
                dynamicSqlData.getTableName(), dynamicSqlData.getWhereParam(), dynamicSqlData.getPageNum(), dynamicSqlData.getPageSize());
        log.info("自定义分页查询参数: {} - {}, 创建sql为:{}", JSON.toJSONString(ds), JSON.toJSONString(dynamicSqlData), pageSelectSql);
        String selectCountSql = SqlHandler.createSelectCountSql(dynamicSqlData.getTableName());
        TableDataInfo data = new TableDataInfo();
        // 切换数据源
        DataSourceUtil.switchDs(ds.getUrl(), ds.getDriverClass(), ds.getUsername(), ds.getPassword(), ds.getDatasourceName());
        List<Map<String, Object>> counts = sysDatasourceMapper.executeSql(selectCountSql);
        data.setTotal(Long.parseLong(String.valueOf(counts.iterator().next().get("COUNT"))));
        List<Map<String, Object>> rowList = Optional.of(sysDatasourceMapper.executeSql(pageSelectSql)).orElse(new ArrayList<>());
        data.setRows(rowList);
        // 恢复默认数据源
        DataSourceUtil.switchDefaultDs();
        return data;
    }

    @Override
    public void saveByParam(DynamicSqlData dynamicSqlData) { // saveBatchByParam
        SysDatasource ds = selectSysDatasourceByDatasourceId(dynamicSqlData.getDatasourceId());

        boolean isOracle = SqlHandler.isOracle(ds.getDriverClass());
        // 切换数据源
        DataSourceUtil.switchDs(ds.getUrl(), ds.getDriverClass(), ds.getUsername(), ds.getPassword(), ds.getDatasourceName());
        if (isOracle
                && null == dynamicSqlData.getPrimaryParam()
                && !SqlHandler.ORACLE_PRIMARY_KEY_ALIAS.equals(dynamicSqlData.getPrimaryKey())) {
            // 仅仅新增的时候，并且有设置主键的时候，查询数据库主键字段对应值是否存在
            MyKeyValue myKeyValue = dynamicSqlData.getDataParam().stream().filter(d -> dynamicSqlData.getPrimaryKey().equals(d.getKey())).findFirst().get();
            String selectSql = String.format("select %s DATA from %s where %s = '%s'",
                    dynamicSqlData.getPrimaryKey(), dynamicSqlData.getTableName(), myKeyValue.getKey(), myKeyValue.getValue());
            List<Map<String, Object>> maps = sysDatasourceMapper.executeSql(selectSql);
            if (!CollectionUtils.isEmpty(maps)) {
                throw new ServiceException("该主键对应数据已存在");
            }
        }

        if (isOracle
                && null != dynamicSqlData.getPrimaryParam()
                && SqlHandler.ORACLE_PRIMARY_KEY_ALIAS.equals(dynamicSqlData.getPrimaryKey())) {
            // 编辑并且没有设置主键的时候，不饿能用别名去更新了，得用rowid
            MyKeyValue myKeyValue = new MyKeyValue();
            myKeyValue.setKey(SqlHandler.ORACLE_PRIMARY_KEY);
            myKeyValue.setValue(dynamicSqlData.getPrimaryParam().getValue());
            dynamicSqlData.setPrimaryParam(myKeyValue);
        }

        String saveSql = null == dynamicSqlData.getPrimaryParam()
                ? SqlHandler.createInsertSql(dynamicSqlData.getTableName(), dynamicSqlData.getDataParam())
                : SqlHandler.createUpdateSql(dynamicSqlData.getTableName(), dynamicSqlData.getDataParam(), dynamicSqlData.getPrimaryParam());
        log.info("自定义保存参数:{} - {} , 创建sql为:{}", JSON.toJSONString(ds), JSON.toJSONString(dynamicSqlData), saveSql);
        sysDatasourceMapper.executeDDL(saveSql);
        // 恢复默认数据源
        DataSourceUtil.switchDefaultDs();
    }

    @Override
    public void saveBatchByParam(List<DynamicSqlData> list, Boolean isCover) {
        if (CollectionUtils.isEmpty(list)) {
            throw new ServiceException("数据为空");
        }
        SysDatasource ds = selectSysDatasourceByDatasourceId(list.iterator().next().getDatasourceId());

        // 切换数据源
        DataSourceUtil.switchDs(ds.getUrl(), ds.getDriverClass(), ds.getUsername(), ds.getPassword(), ds.getDatasourceName());

        try {
            sysDatasourceServiceTransactional.saveBatch(list, isCover, ds);
        } catch (Exception e) {
            throw e;
        } finally {
            // 恢复默认数据源
            DataSourceUtil.switchDefaultDs();
        }
    }

    @Override
    public void deleteByParam(DynamicSqlData dynamicSqlData) {
        SysDatasource ds = selectSysDatasourceByDatasourceId(dynamicSqlData.getDatasourceId());
        // 切换数据源
        DataSourceUtil.switchDs(ds.getUrl(), ds.getDriverClass(), ds.getUsername(), ds.getPassword(), ds.getDatasourceName());

        MyKeyValue primaryParam = dynamicSqlData.getPrimaryParam();
        if (primaryParam.getKey().equals(SqlHandler.ORACLE_PRIMARY_KEY_ALIAS)) {
            primaryParam.setKey(SqlHandler.ORACLE_PRIMARY_KEY);
        }
        String sql = String.format("delete from %s where %s = '%s'", dynamicSqlData.getTableName(), primaryParam.getKey(), primaryParam.getValue());
        log.info("动态删除数据:{}", JSON.toJSONString(dynamicSqlData));
        sysDatasourceMapper.executeDDL(sql);
        // 恢复默认数据源
        DataSourceUtil.switchDefaultDs();
    }

    /**
     * 是否为查询操作
     *
     * @param sql
     * @return
     */
    private boolean isSelect(String sql) {
        return sql.trim().toLowerCase().startsWith("select");
    }

    //获取字符编码字典值
    private Charset getCharset(String charsetValue) {
        return CharsetKit.charset(DictUtils.getDictLabel("sys_charset", charsetValue));
    }

    //切换数据源
    private void switchDsByDsId(Long dsId) {
        SysDatasource ds = selectSysDatasourceByDatasourceId(dsId);
        if (StringUtils.isNotNull(ds)) {
            DataSourceUtil.switchDs(ds.getUrl(), ds.getDriverClass(), ds.getUsername(), ds.getPassword(), ds.getDatasourceName());
//            DataSourceUtil.switchDs(ds.getDatasourceName());
        } else {
            throw new ServiceException("未找到该数据源");
        }

    }

    @Override
    public void executeDDL(Long datasourceId, String sql) {
        switchDsByDsId(datasourceId);
        sysDatasourceMapper.executeDDL(sql);
        DataSourceUtil.switchDefaultDs();
    }
}
