package com.tbyf.his.web.controller.dataImport;

import com.alibaba.druid.pool.DruidDataSource;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.tbyf.his.common.annotation.IgnoreWebSecurity;
import com.tbyf.his.common.core.domain.AjaxResult;
import com.tbyf.his.common.core.page.TableDataInfo;
import com.tbyf.his.common.core.text.StrFormatter;
import com.tbyf.his.common.utils.StringUtils;
import com.tbyf.his.common.utils.bean.BeanUtils;
import com.tbyf.his.framework.datasource.DataSourceUtil;
import com.tbyf.his.framework.datasource.DynamicDataSourceContextHolder;
import com.tbyf.his.system.domain.SysDatasource;
import com.tbyf.his.system.service.ISysDatasourceService;
import com.tbyf.his.web.dataImport.domain.param.*;
import com.tbyf.his.web.dataImport.entity.DataDict;
import com.tbyf.his.web.dataImport.entity.DataField;
import com.tbyf.his.web.dataImport.entity.DataImportTemplate;
import com.tbyf.his.web.dataImport.entity.DataRule;
import com.tbyf.his.web.dataImport.service.DataDictService;
import com.tbyf.his.web.dataImport.service.DataFieldService;
import com.tbyf.his.web.dataImport.service.DataImportTemplateService;
import com.tbyf.his.web.dataImport.service.DataRuleService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;

/**
 * @author lzz
 * @date 2023/2/7 10:42
 */
@RestController
@Api(tags = "数据导入接口")
@RequestMapping("/dataImport")
@Slf4j
public class DataImportController {

    @Autowired
    private DataImportTemplateService dataImportService;

    @Autowired
    private DataDictService dictService;

    @Autowired
    private ISysDatasourceService sysDatasourceService;

    @Autowired
    private JdbcTemplate jdbcTemplate;

    @Autowired
    private DataFieldService dataFieldService;

    @Autowired
    private DataRuleService dataRuleService;

    @IgnoreWebSecurity
    @GetMapping("/template")
    @ApiOperation("模板查询")
    public AjaxResult queryTemplate(@Validated QueryTemplateParam param) {
        final LambdaQueryWrapper<DataImportTemplate> wrapper = Wrappers.lambdaQuery(DataImportTemplate.class)
                .eq(DataImportTemplate::getYear, param.getYear());
        return AjaxResult.success(dataImportService.list(wrapper));
    }

    @IgnoreWebSecurity
    @PostMapping("/template")
    @ApiOperation("模板新增")
    public AjaxResult addTemplate(@RequestBody @Validated AddTemplateParam param) {
        DataImportTemplate template = new DataImportTemplate();
        BeanUtils.copyProperties(param, template);
        template.initAdd();
        dataImportService.save(template);
        return AjaxResult.success();
    }

    @IgnoreWebSecurity
    @PutMapping("/template")
    @ApiOperation("修改模板")
    public AjaxResult updateTemplate(@RequestBody @Validated UpdateTemplateParam param) {
        DataImportTemplate template = new DataImportTemplate();
        BeanUtils.copyProperties(param, template);
        template.initAdd();
        dataImportService.updateById(template);
        return AjaxResult.success();
    }

    @IgnoreWebSecurity
    @DeleteMapping("/template")
    @ApiOperation("删除模板")
    public AjaxResult deleteTemplate(@RequestParam String templateId) {
        dataImportService.removeById(templateId);
        //TODO 需要删除字段以及其字段关联的规则与其它数据
        dataFieldService.remove(Wrappers.lambdaQuery(DataField.class).eq(DataField::getTemplateId, templateId));
        return AjaxResult.success();
    }

    @IgnoreWebSecurity
    @GetMapping("/dict")
    @ApiOperation("字典查询")
    public AjaxResult queryDict(@RequestParam String type) {
        final LambdaQueryWrapper<DataDict> wrapper = Wrappers.lambdaQuery(DataDict.class)
                .eq(DataDict::getType, type);
        final List<Map<String, String>> list = dictService.list(wrapper).stream().map(item -> {
            String json = item.getRemarks();
            item.setRemarks(null);
            final Map<String, String> map = BeanUtils.getFieldValueMap(item);
            if (StringUtils.isNotBlank(json)) {
                try {
                    final JSONObject object = JSON.parseObject(json);
                    object.forEach((key, value) -> {
                        map.put(key, (String) value);
                    });
                } catch (Exception ignore) {
                }
            }
            return map;
        }).collect(Collectors.toList());
        return AjaxResult.success(list);
    }

    @IgnoreWebSecurity
    @PostMapping("/dict")
    @ApiOperation("新增字典")
    public AjaxResult addDict(@RequestBody AddDictParam param) {
        DataDict dict = new DataDict();
        BeanUtils.copyProperties(param, dict);
        dictService.save(dict);
        return AjaxResult.success();
    }

    @IgnoreWebSecurity
    @PutMapping("/dict")
    @ApiOperation("修改字典")
    public AjaxResult updateDict(@RequestBody @Validated UpdateDictParam param) {
        DataDict template = new DataDict();
        BeanUtils.copyProperties(param, template);
        dictService.updateById(template);
        return AjaxResult.success();
    }

    @IgnoreWebSecurity
    @DeleteMapping("/dict")
    @ApiOperation("删除字典")
    public AjaxResult deleteDict(@RequestParam String dictId) {
        dictService.removeById(dictId);
        return AjaxResult.success();
    }

    @IgnoreWebSecurity
    @GetMapping("/datasource")
    @ApiOperation("查询所有数据源")
    public AjaxResult queryDatasource() {
        final List<SysDatasource> list = sysDatasourceService.selectSysDatasourceList(new SysDatasource());
        return AjaxResult.success(list);
    }

    @IgnoreWebSecurity
    @GetMapping("/datasource/table")
    @ApiOperation("获取指定数据源中的表")
    public AjaxResult getDbTable(@RequestParam String dataSourceId) {
        try {
            DataSourceUtil.switchDs(dataSourceId);
            final DruidDataSource dataSource = (DruidDataSource) DynamicDataSourceContextHolder.dataSourcesMap.get(dataSourceId);
            String sql = "select  TABLE_NAME AS VALUE,COMMENTS AS LABEL from user_tab_comments WHERE TABLE_TYPE = 'TABLE'";
            if (dataSource.getDriverClassName().toLowerCase().contains("mysql")) {
                sql = "select  TABLE_NAME AS VALUE ,table_comment AS LABEL from information_schema.tables where  table_type='BASE TABLE'";
            }
            final List<Map<String, Object>> maps = jdbcTemplate.queryForList(sql);
            return AjaxResult.success(maps.stream().filter(
                    distinctByKey(b -> b.get("VALUE"))
            ).distinct().collect(Collectors.toList()));
        } finally {
            DataSourceUtil.switchDefaultDs();
        }
    }

    @IgnoreWebSecurity
    @GetMapping("/datasource/field")
    @ApiOperation("获取指定数据表中的字段")
    public AjaxResult getDbField(@RequestParam String dataSourceId, @RequestParam String tableName) {
        try {
            DataSourceUtil.switchDs(dataSourceId);
            final DruidDataSource dataSource = (DruidDataSource) DynamicDataSourceContextHolder.dataSourcesMap.get(dataSourceId);
            String sql = StrFormatter.format("SELECT UTC.COLUMN_NAME AS VALUE,UTC.DATA_TYPE AS TYPE,( SELECT  UCC.COMMENTS  FROM user_col_comments UCC WHERE UCC.COLUMN_NAME = UTC.COLUMN_NAME AND UCC.TABLE_NAME = UTC.TABLE_NAME ) AS LABEL FROM user_tab_columns UTC WHERE UTC.TABLE_NAME = '{}'", tableName);
            if (dataSource.getDriverClassName().toLowerCase().contains("mysql")) {
                sql = StrFormatter.format("SELECT column_name AS VALUE, DATA_TYPE AS TYPE, column_comment AS LABEL FROM information_schema.COLUMNS WHERE table_name = '{}'", tableName);
            }
            final List<Map<String, Object>> maps = jdbcTemplate.queryForList(sql);
            return AjaxResult.success(maps);
        } finally {
            DataSourceUtil.switchDefaultDs();
        }
    }


    public static <T> Predicate<T> distinctByKey(Function<? super T, ?> keyExtractor) {
        Map<Object, Boolean> seen = new ConcurrentHashMap<>();
        return t -> seen.putIfAbsent(keyExtractor.apply(t), Boolean.TRUE) == null;
    }


    @IgnoreWebSecurity
    @GetMapping("/field")
    @ApiOperation("字段查询")
    public TableDataInfo queryField(@Validated QueryFieldParam param) {
        final Page<DataField> page = Page.of(param.getPageNum(), param.getPageSize());
        final LambdaQueryWrapper<DataField> queryWrapper = Wrappers.lambdaQuery(DataField.class);
        queryWrapper.eq(StringUtils.isNotBlank(param.getTemplateId()), DataField::getTemplateId, param.getTemplateId())
                .like(StringUtils.isNotBlank(param.getCode()), DataField::getCode, param.getCode())
                .like(StringUtils.isNotBlank(param.getTitle()), DataField::getTitle, param.getTitle())
                .like(StringUtils.isNotBlank(param.getCoordinate()), DataField::getCoordinate, param.getCoordinate())
                .like(StringUtils.isNotBlank(param.getField()), DataField::getField, param.getField());
        final Page<DataField> templatePage = dataFieldService.page(page, queryWrapper);
        return param.convert(templatePage);
    }

    @IgnoreWebSecurity
    @PostMapping("/field")
    @ApiOperation("字段新增")
    public AjaxResult addField(@RequestBody @Validated AddFieldParam param) {
        DataField field = new DataField();
        BeanUtils.copyProperties(param, field);
        dataFieldService.save(field);
        return AjaxResult.success();
    }

    @IgnoreWebSecurity
    @PutMapping("/field")
    @ApiOperation("修改字段")
    public AjaxResult updateField(@RequestBody @Validated UpdateFieldParam param) {
        DataField field = new DataField();
        BeanUtils.copyProperties(param, field);
        dataFieldService.updateById(field);
        return AjaxResult.success();
    }

    @IgnoreWebSecurity
    @DeleteMapping("/field")
    @ApiOperation("删除字段")
    public AjaxResult deleteField(@RequestParam String fieldId) {
        dataFieldService.removeById(fieldId);
        //TODO 需要删除其字段关联的规则与其它数据
        return AjaxResult.success();
    }

    @IgnoreWebSecurity
    @GetMapping("/rule")
    @ApiOperation("规则查询")
    public TableDataInfo queryRule(@Validated QueryRuleParam param) {
        final Page<DataRule> page = Page.of(param.getPageNum(), param.getPageSize());
        final LambdaQueryWrapper<DataRule> queryWrapper = Wrappers.lambdaQuery(DataRule.class);
        queryWrapper.like(StringUtils.isNotBlank(param.getName()), DataRule::getName, param.getName())
                .eq(StringUtils.isNotBlank(param.getType()), DataRule::getType, param.getType())
                .in(DataRule::getType, "基础规则", "组合规则");
        final Page<DataRule> templatePage = dataRuleService.page(page, queryWrapper);
        return param.convert(templatePage);
    }

    @IgnoreWebSecurity
    @GetMapping("/rule/dict")
    @ApiOperation("规则字典")
    public AjaxResult queryRuleDict() {
        final LambdaQueryWrapper<DataRule> queryWrapper = Wrappers.lambdaQuery(DataRule.class);
        queryWrapper.in(DataRule::getType, "基础规则", "组合规则")
                .select(DataRule::getId, DataRule::getName);
        return AjaxResult.success(dataRuleService.list(queryWrapper));
    }

    @IgnoreWebSecurity
    @PostMapping("/rule")
    @ApiOperation("规则新增")
    public AjaxResult addRule(@RequestBody @Validated AddRuleParam param) {
        DataRule rule = new DataRule();
        BeanUtils.copyProperties(param, rule);
        dataRuleService.save(rule);
        return AjaxResult.success();
    }

    @IgnoreWebSecurity
    @PutMapping("/rule")
    @ApiOperation("修改规则")
    public AjaxResult updateRule(@RequestBody @Validated UpdateRuleParam param) {
        DataRule rule = new DataRule();
        BeanUtils.copyProperties(param, rule);
        dataRuleService.updateById(rule);
        return AjaxResult.success();
    }

    @IgnoreWebSecurity
    @DeleteMapping("/rule")
    @ApiOperation("删除规则")
    public AjaxResult deleteRule(@RequestParam String ruleId) {
        dataRuleService.removeById(ruleId);
        return AjaxResult.success();
    }

}
