package com.tbyf.his.web.controller.dataImport;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.tbyf.his.common.annotation.IgnoreWebSecurity;
import com.tbyf.his.common.core.domain.AjaxResult;
import com.tbyf.his.common.core.page.TableDataInfo;
import com.tbyf.his.common.utils.StringUtils;
import com.tbyf.his.common.utils.bean.BeanUtils;
import com.tbyf.his.web.dataImport.core.DiConfig;
import com.tbyf.his.web.dataImport.domain.param.AddFieldParam;
import com.tbyf.his.web.dataImport.domain.param.QueryFieldParam;
import com.tbyf.his.web.dataImport.domain.param.UpdateFieldParam;
import com.tbyf.his.web.dataImport.entity.BindRule;
import com.tbyf.his.web.dataImport.entity.DataField;
import com.tbyf.his.web.dataImport.entity.ExcelData;
import com.tbyf.his.web.dataImport.service.BindRuleService;
import com.tbyf.his.web.dataImport.service.DataFieldService;
import com.tbyf.his.web.dataImport.service.ExcelDataService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.ss.usermodel.WorkbookFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.CollectionUtils;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

/**
 * @author lzz
 * @date 2023/2/16 15:29
 */
@RestController
@Api(tags = "数据字段接口")
@RequestMapping("/data/field")
@Slf4j
public class DataFieldController {

    @Autowired
    private DataFieldService dataFieldService;

    @Autowired
    private BindRuleService bindRuleService;

    @Autowired
    private ExcelDataService excelDataService;

    @IgnoreWebSecurity
    @GetMapping("")
    @ApiOperation("字段查询")
    public TableDataInfo queryField(@Validated QueryFieldParam param) {
        final Page<DataField> page = Page.of(param.getPageNum(), param.getPageSize());
        final LambdaQueryWrapper<DataField> queryWrapper = Wrappers.lambdaQuery(DataField.class);
        queryWrapper.eq(StringUtils.isNotBlank(param.getTemplateId()), DataField::getTemplateId, param.getTemplateId())
                .like(StringUtils.isNotBlank(param.getCode()), DataField::getCode, param.getCode())
                .like(StringUtils.isNotBlank(param.getTitle()), DataField::getTitle, param.getTitle())
                .like(StringUtils.isNotBlank(param.getCoordinate()), DataField::getCoordinate, param.getCoordinate())
                .like(StringUtils.isNotBlank(param.getField()), DataField::getField, param.getField());
        final Page<DataField> templatePage = dataFieldService.page(page, queryWrapper);
        return param.convert(templatePage);
    }

    @IgnoreWebSecurity
    @PostMapping("")
    @ApiOperation("字段新增")
    public AjaxResult addField(@RequestBody @Validated AddFieldParam param) {
        DataField field = new DataField();
        BeanUtils.copyProperties(param, field);
        dataFieldService.save(field);
        return AjaxResult.success();
    }

    @IgnoreWebSecurity
    @PostMapping("/update")
    @ApiOperation("修改字段")
    public AjaxResult updateField(@RequestBody @Validated UpdateFieldParam param) {
        DataField field = new DataField();
        BeanUtils.copyProperties(param, field);
        dataFieldService.updateById(field);
        return AjaxResult.success();
    }

    @IgnoreWebSecurity
    @GetMapping("/delete")
    @ApiOperation("删除字段")
    public AjaxResult deleteField(@RequestParam String fieldId) {
        dataFieldService.removeById(fieldId);
        bindRuleService.remove(Wrappers.lambdaQuery(BindRule.class).eq(BindRule::getDataId, fieldId));
        return AjaxResult.success();
    }

    @IgnoreWebSecurity
    @GetMapping("/reset")
    @ApiOperation("根据基础模板重置字段")
    public AjaxResult resetField(@RequestParam String excelId) {
        final ExcelData excelData = excelDataService.getById(excelId);
        final LambdaQueryWrapper<DataField> wrapper = Wrappers.lambdaQuery(DataField.class).eq(DataField::getTemplateId, excelData.getTemplateId());
        final List<DataField> list = dataFieldService.list(wrapper);
        if (!CollectionUtils.isEmpty(list)) {
            final List<String> fieldIdList = list.stream().map(DataField::getId).collect(Collectors.toList());
            bindRuleService.remove(Wrappers.lambdaQuery(BindRule.class).in(BindRule::getDataId, fieldIdList));
        }
        dataFieldService.remove(Wrappers.lambdaQuery(DataField.class).eq(DataField::getTemplateId, excelData.getTemplateId()));
        try (InputStream is = new ByteArrayInputStream(excelData.getFile()); Workbook workbook = WorkbookFactory.create(is)) {
            final Sheet sheet = workbook.getSheetAt(0);
            final int rows = sheet.getLastRowNum() + 1;
            List<DataField> fieldList = new ArrayList<>();
            for (int i = 1; i < rows; i++) {
                final Row row = sheet.getRow(i);
                String code = DiConfig.getValue(row.getCell(1, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL));
                String title = DiConfig.getValue(row.getCell(2, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL));
                String unit = DiConfig.getValue(row.getCell(4, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL));
                if (StringUtils.isAllBlank(code, title, unit)) {
                    continue;
                }
                if (StringUtils.equals(code, "代码") || StringUtils.equals(title, "指标名称") || StringUtils.equals(unit, "计量单位")) {
                    continue;
                }
                final DataField dataField = new DataField();
                dataField.setTemplateId(excelData.getTemplateId());
                dataField.setCode(code);
                dataField.setTitle(title);
                dataField.setUnit(unit);
                // TODO 这里坐标修改为i+1是因为excel文件中展示的第一行号是1,符合直觉
                dataField.setCoordinate("F," + (i + 1));
                dataField.setSort(i + 1);
                dataField.createField();
                fieldList.add(dataField);
            }
            dataFieldService.saveBatch(fieldList);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
        return AjaxResult.success();
    }

}
