package com.tbyf.his.web.controller.dataImport;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.tbyf.his.common.annotation.IgnoreWebSecurity;
import com.tbyf.his.common.core.domain.AjaxResult;
import com.tbyf.his.common.core.page.TableDataInfo;
import com.tbyf.his.common.utils.StringUtils;
import com.tbyf.his.common.utils.bean.BeanUtils;
import com.tbyf.his.web.dataImport.core.DiConfig;
import com.tbyf.his.web.dataImport.core.RuleVO;
import com.tbyf.his.web.dataImport.core.RuleValidator;
import com.tbyf.his.web.dataImport.domain.param.AddRuleParam;
import com.tbyf.his.web.dataImport.domain.param.BindRuleParam;
import com.tbyf.his.web.dataImport.domain.param.QueryRuleParam;
import com.tbyf.his.web.dataImport.domain.param.UpdateRuleParam;
import com.tbyf.his.web.dataImport.entity.BindRule;
import com.tbyf.his.web.dataImport.entity.DataField;
import com.tbyf.his.web.dataImport.entity.DataRule;
import com.tbyf.his.web.dataImport.service.BindRuleService;
import com.tbyf.his.web.dataImport.service.DataFieldService;
import com.tbyf.his.web.dataImport.service.DataRuleService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

/**
 * @author lzz
 * @date 2023/2/16 15:29
 */
@RestController
@Api(tags = "数据字段接口")
@RequestMapping("/data/rule")
@Slf4j
public class DataRuleController {

    @Autowired
    private DataRuleService dataRuleService;

    @Autowired
    private BindRuleService bindRuleService;

    @Autowired
    private DataFieldService dataFieldService;

    @IgnoreWebSecurity
    @GetMapping("")
    @ApiOperation("规则查询")
    public TableDataInfo queryRule(@Validated QueryRuleParam param) {
        final Page<DataRule> page = Page.of(param.getPageNum(), param.getPageSize());
        final LambdaQueryWrapper<DataRule> queryWrapper = Wrappers.lambdaQuery(DataRule.class);
        queryWrapper.like(StringUtils.isNotBlank(param.getName()), DataRule::getName, param.getName())
                .eq(StringUtils.isNotBlank(param.getType()), DataRule::getType, param.getType())
                .in(DataRule::getType, "基础规则", "组合规则");
        final Page<DataRule> templatePage = dataRuleService.page(page, queryWrapper);
        return param.convert(templatePage);
    }

    @IgnoreWebSecurity
    @GetMapping("/dict")
    @ApiOperation("规则字典")
    public AjaxResult queryRuleDict() {
        final LambdaQueryWrapper<DataRule> queryWrapper = Wrappers.lambdaQuery(DataRule.class);
        queryWrapper.in(DataRule::getType, "基础规则", "组合规则")
                .select(DataRule::getId, DataRule::getName);
        return AjaxResult.success(dataRuleService.list(queryWrapper));
    }

    @IgnoreWebSecurity
    @PostMapping("")
    @ApiOperation("规则新增")
    public AjaxResult addRule(@RequestBody @Validated AddRuleParam param) {
        DataRule rule = new DataRule();
        BeanUtils.copyProperties(param, rule);
        dataRuleService.save(rule);
        return AjaxResult.success();
    }

    @IgnoreWebSecurity
    @PostMapping("/update")
    @ApiOperation("修改规则")
    public AjaxResult updateRule(@RequestBody @Validated UpdateRuleParam param) {
        DataRule rule = new DataRule();
        BeanUtils.copyProperties(param, rule);
        dataRuleService.updateById(rule);
        return AjaxResult.success();
    }

    @IgnoreWebSecurity
    @GetMapping("/delete")
    @ApiOperation("删除规则")
    public AjaxResult deleteRule(@RequestParam String ruleId) {
        dataRuleService.removeById(ruleId);
        bindRuleService.remove(Wrappers.lambdaQuery(BindRule.class).eq(BindRule::getRuleId, ruleId));
        return AjaxResult.success();
    }

    @IgnoreWebSecurity
    @PostMapping("/test")
    @ApiOperation("规则测试")
    public AjaxResult testRule(@RequestBody @Validated RuleVO test) {
        final RuleValidator validator = DiConfig.getValidator(test.getMode());
        validator.validate(test);
        if (StringUtils.isNotBlank(test.getResult())) {
            return AjaxResult.error(test.getResult());
        }
        return AjaxResult.success();
    }

    @IgnoreWebSecurity
    @GetMapping("/bind/quick")
    @ApiOperation("快速绑定")
    public AjaxResult quickBind(@RequestParam String templateId,
                                @RequestParam String ruleId) {
        final LambdaQueryWrapper<DataField> queryWrapper = Wrappers.lambdaQuery(DataField.class)
                .select(DataField::getId)
                .eq(DataField::getTemplateId, templateId);
        final List<String> fieldIdList = dataFieldService.list(queryWrapper).stream()
                .map(DataField::getId)
                .collect(Collectors.toList());
        final LambdaQueryWrapper<BindRule> removeWrapper = Wrappers.lambdaQuery(BindRule.class)
                .in(BindRule::getDataId, fieldIdList)
                .eq(BindRule::getRuleId, ruleId);
        bindRuleService.remove(removeWrapper);
        final List<BindRule> bindRuleList = fieldIdList.stream().map(item -> {
            BindRule bind = new BindRule();
            bind.setDataId(item);
            bind.setRuleId(ruleId);
            return bind;
        }).collect(Collectors.toList());
        bindRuleService.saveBatch(bindRuleList);
        return AjaxResult.success();
    }

    @IgnoreWebSecurity
    @GetMapping("/bind")
    @ApiOperation("查询绑定的规则")
    public AjaxResult queryBindRule(@RequestParam String fieldId) {
        final List<BindRule> list = bindRuleService.list(Wrappers.lambdaQuery(BindRule.class)
                .eq(BindRule::getDataId, fieldId));
        if (CollectionUtils.isEmpty(list)) {
            return AjaxResult.success(Collections.emptyList());
        }
        final List<String> idList = list.stream().map(BindRule::getRuleId).collect(Collectors.toList());
        return AjaxResult.success(dataRuleService.list(Wrappers.lambdaQuery(DataRule.class)
                .in(DataRule::getId, idList)));
    }

    @IgnoreWebSecurity
    @PostMapping("/bind")
    @ApiOperation("配置字段规则绑定")
    @Transactional(rollbackFor = Exception.class)
    public AjaxResult bindRule(@RequestBody BindRuleParam param) {
        bindRuleService.remove(Wrappers.lambdaQuery(BindRule.class).eq(BindRule::getDataId, param.getFieldId()));
        if (!CollectionUtils.isEmpty(param.getRuleIdList())) {
            final List<BindRule> bindRuleList = param.getRuleIdList().stream().map(item -> {
                BindRule bd = new BindRule();
                bd.setRuleId(item);
                bd.setDataId(param.getFieldId());
                return bd;
            }).collect(Collectors.toList());
            bindRuleService.saveBatch(bindRuleList);
        }
        return AjaxResult.success();
    }

}
