package com.tbyf.his.web.controller.dataImport;

import com.alibaba.druid.pool.DruidDataSource;
import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.annotation.DbType;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.tbyf.his.common.annotation.IgnoreWebSecurity;
import com.tbyf.his.common.core.domain.AjaxResult;
import com.tbyf.his.common.core.text.StrFormatter;
import com.tbyf.his.common.exception.base.BaseException;
import com.tbyf.his.common.utils.StringUtils;
import com.tbyf.his.framework.datasource.DataSourceUtil;
import com.tbyf.his.framework.datasource.DynamicDataSourceContextHolder;
import com.tbyf.his.system.domain.SysDatasource;
import com.tbyf.his.system.service.ISysDatasourceService;
import com.tbyf.his.system.vo.SqlHandler;
import com.tbyf.his.web.dataImport.domain.vo.TemplateVO;
import com.tbyf.his.web.dataImport.entity.DataField;
import com.tbyf.his.web.dataImport.entity.DataTemplate;
import com.tbyf.his.web.dataImport.entity.ExcelData;
import com.tbyf.his.web.dataImport.service.DataFieldService;
import com.tbyf.his.web.dataImport.service.DataSourceService;
import com.tbyf.his.web.dataImport.service.DataTemplateService;
import com.tbyf.his.web.dataImport.service.ExcelDataService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.util.CollectionUtils;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;

/**
 * @author lzz
 * @date 2023/2/7 10:42
 */
@RestController
@Api(tags = "数据导入接口")
@RequestMapping("/data/import")
@Slf4j
public class DataImportController {

    @Autowired
    private DataTemplateService dataImportService;
    @Autowired
    private ISysDatasourceService sysDatasourceService;

    @Autowired
    private JdbcTemplate jdbcTemplate;

    @Autowired
    private DataFieldService dataFieldService;

    @Autowired
    private ExcelDataService excelDataService;

    @IgnoreWebSecurity
    @GetMapping("/datasource")
    @ApiOperation("查询所有数据源")
    public AjaxResult queryDatasource() {
        final List<SysDatasource> list = sysDatasourceService.selectSysDatasourceList(new SysDatasource());
        return AjaxResult.success(list);
    }

    @IgnoreWebSecurity
    @GetMapping("/datasource/table")
    @ApiOperation("获取指定数据源中的表")
    public AjaxResult getDbTable(@RequestParam String dataSourceId) {
        try {
            DataSourceService.switchDb(dataSourceId);
            final DruidDataSource dataSource = (DruidDataSource) DynamicDataSourceContextHolder.dataSourcesMap.get(dataSourceId);
            String sql;
            if (SqlHandler.isOracle(dataSource.getDriverClassName())) {
                sql = "SELECT TABLE_NAME AS VALUE,COMMENTS AS LABEL FROM USER_TAB_COMMENTS WHERE TABLE_TYPE = 'TABLE'";
            } else {
                sql = "select TABLE_NAME  AS VALUE,TABLE_COMMENT AS LABEL from information_schema.tables where table_type='BASE TABLE'";
            }
            final List<Map<String, Object>> maps = jdbcTemplate.queryForList(sql);
            return AjaxResult.success(maps.stream().filter(
                    distinctByKey(b -> b.get("VALUE"))
            ).distinct().collect(Collectors.toList()));
        } finally {
            DataSourceService.switchDefault();
        }
    }

    @IgnoreWebSecurity
    @GetMapping("/datasource/field")
    @ApiOperation("获取指定数据表中的字段")
    public AjaxResult getDbField(@RequestParam String dataSourceId, @RequestParam String tableName) {
        try {
            DataSourceUtil.switchDs(dataSourceId);
            final DruidDataSource dataSource = (DruidDataSource) DynamicDataSourceContextHolder.dataSourcesMap.get(dataSourceId);
            String sql = StrFormatter.format("SELECT UTC.COLUMN_NAME AS VALUE,UTC.DATA_TYPE AS TYPE,( SELECT  UCC.COMMENTS  FROM user_col_comments UCC WHERE UCC.COLUMN_NAME = UTC.COLUMN_NAME AND UCC.TABLE_NAME = UTC.TABLE_NAME ) AS LABEL FROM user_tab_columns UTC WHERE UTC.TABLE_NAME = '{}'", tableName);
            if (dataSource.getDriverClassName().toLowerCase().contains("mysql")) {
                sql = StrFormatter.format("SELECT column_name AS VALUE, DATA_TYPE AS TYPE, column_comment AS LABEL FROM information_schema.COLUMNS WHERE table_name = '{}'", tableName);
            }
            final List<Map<String, Object>> maps = jdbcTemplate.queryForList(sql);
            return AjaxResult.success(maps);
        } finally {
            DataSourceUtil.switchDefaultDs();
        }
    }


    public static <T> Predicate<T> distinctByKey(Function<? super T, ?> keyExtractor) {
        Map<Object, Boolean> seen = new ConcurrentHashMap<>();
        return t -> seen.putIfAbsent(keyExtractor.apply(t), Boolean.TRUE) == null;
    }


    @IgnoreWebSecurity
    @GetMapping("/table/create")
    @ApiOperation("物理表生成")
    public AjaxResult createTable(@RequestParam String tableName,
                                  @RequestParam String templateId) {
        // 默认表名全大写
        tableName = tableName.toUpperCase();
        final DataTemplate template = dataImportService.getById(templateId);
        if (StringUtils.isBlank(template.getDataSourceId())) {
            return AjaxResult.error("请在模板编辑界面选择对应的数据源");
        }
        final List<DataField> fieldList = dataFieldService.list(Wrappers.lambdaQuery(DataField.class)
                .isNotNull(DataField::getField)
                .eq(DataField::getTemplateId, template.getId())
                .orderByAsc(DataField::getSort));
        if (CollectionUtils.isEmpty(fieldList)) {
            return AjaxResult.error("没有可生成表的字段配置,请新建字段");
        }
        try {
            final DbType dbType = dataImportService.getDbType(template.getDataSourceId());
            DataSourceUtil.switchDs(template.getDataSourceId());
            if (dbType.equals(DbType.MYSQL)) {
                // mysql
                final Integer count = jdbcTemplate.queryForObject(StrFormatter.format("select count(table_name) from information_schema.TABLES where TABLE_NAME = '{}'", tableName), Integer.class);
                if (count > 0) {
                    return AjaxResult.error("此表已存在");
                }
                StringBuilder sb = new StringBuilder();
                sb.append("create table {} ( ");
                fieldList.forEach(field -> sb.append(field.getField())
                        .append(" varchar(32) null comment '")
                        .append(field.createComment()).append("',"));
                sb.append(" YEAR varchar(32) null").append(" ) comment '")
                        .append(template.createComment())
                        .append("';");
                // 同时创建数据表与临时表
                log.info("生成物理表的sql为:{}", sb);
                jdbcTemplate.execute(StrFormatter.format(sb.toString(), tableName));
                jdbcTemplate.execute(StrFormatter.format(sb.toString(), tableName + "_TEMP"));
                // 修改模板表绑定的表名
                final DataTemplate updateTemplate = new DataTemplate();
                updateTemplate.setId(template.getId());
                updateTemplate.setTableName(tableName);
                DataSourceUtil.switchDefaultDs();
                dataImportService.updateById(updateTemplate);
            } else {
                // oracle
                final Integer count = jdbcTemplate.queryForObject(StrFormatter.format("select count(*) from user_tables where table_name =upper('{}')", tableName), Integer.class);
                if (count > 0) {
                    return AjaxResult.error("此表已存在");
                }
                List<String> prodSqlList = new ArrayList<>();
                List<String> tempSqlList = new ArrayList<>();
                final String tempTableName = tableName + "_TEMP";
                String comment = "COMMENT ON COLUMN {}.{} IS '{}'";
                StringBuilder sql = new StringBuilder();
                sql.append("CREATE TABLE {} ( ");
                String finalTableName = tableName;
                fieldList.forEach(field -> {
                    sql.append(field.getField())
                            .append(" VARCHAR2(255),");
                    final String comm = field.createComment();
                    prodSqlList.add(StrFormatter.format(comment, finalTableName, field.getField(), comm));
                    tempSqlList.add(StrFormatter.format(comment, tempTableName, field.getField(), comm));
                });
                sql.append(" YEAR VARCHAR2(255)")
                        .append(" )");
                prodSqlList.add(0, StrFormatter.format(sql.toString(), finalTableName));
                tempSqlList.add(0, StrFormatter.format(sql.toString(), tempTableName));
                prodSqlList.add(StrFormatter.format("COMMENT ON TABLE {} IS '{}'", finalTableName, template.createComment()));
                tempSqlList.add(StrFormatter.format("COMMENT ON TABLE {} IS '{}'", tempTableName, template.createComment()));
                // 同时创建数据表与临时表
                String[] prod = new String[prodSqlList.size()];
                String[] temp = new String[tempSqlList.size()];
                prodSqlList.toArray(prod);
                tempSqlList.toArray(temp);
                /*prodSqlList.forEach(sql1->{
                    jdbcTemplate.execute(sql1);
                });
                tempSqlList.forEach(sql2->{
                    jdbcTemplate.execute(sql2);
                });*/
                jdbcTemplate.batchUpdate(prod);
                jdbcTemplate.batchUpdate(temp);
                // 修改模板表绑定的表名
                final DataTemplate updateTemplate = new DataTemplate();
                updateTemplate.setId(template.getId());
                updateTemplate.setTableName(finalTableName);
                DataSourceUtil.switchDefaultDs();
                dataImportService.updateById(updateTemplate);
            }
            return AjaxResult.success();
        } catch (Exception e) {
            log.error("物理表生成失败", e);
            return AjaxResult.error(e.getMessage());
        } finally {
            DataSourceUtil.switchDefaultDs();
        }
    }

    @IgnoreWebSecurity
    @GetMapping("/analyze/export")
    @ApiOperation("数据分析并导出")
    public void analyzeExport(@RequestParam String templateId,
                              HttpServletResponse response) {
        ExcelData excelData = excelDataService.getOne(Wrappers.lambdaQuery(ExcelData.class)
                .eq(ExcelData::getTemplateId, templateId)
                .eq(ExcelData::getType, "1"), false);
        if (excelData == null) {
            try {
                response.setContentType("application/json");
                response.setCharacterEncoding("UTF-8");
                response.getWriter().println(JSON.toJSONString(AjaxResult.error("请先导入数据文件")));
                response.getWriter().flush();
            } catch (IOException e) {
                throw new RuntimeException(e);
            }
            return;
        }
        excelDataService.analyzeExport(excelData, response);
    }

    @IgnoreWebSecurity
    @PostMapping("/clearTemp")
    @ApiOperation("清空临时表数据")
    public AjaxResult clearTemp(@RequestBody TemplateVO vo) {
        final DataTemplate template = dataImportService.getById(vo.getId());
        try {
            DataSourceUtil.switchDs(template.getDataSourceId());
            String sql = "DELETE FROM " + template.getTableName() + "_TEMP";
            //sql =  sql + " WHERE YEAR = '"+template.getYear()+"'";
            log.info("执行的sql为:[{}]", sql);
            jdbcTemplate.execute(sql);
        } catch (Exception e) {
            throw new RuntimeException(e);
        } finally {
            DataSourceUtil.switchDefaultDs();
        }
        excelDataService.remove(Wrappers.lambdaQuery(ExcelData.class)
                .eq(ExcelData::getType, "1")
                .eq(ExcelData::getTemplateId, template.getId()));
        return AjaxResult.success();
    }

    @IgnoreWebSecurity
    @PostMapping("/syncDb")
    @ApiOperation("一键导入正式库")
    public AjaxResult syncDb(@RequestBody TemplateVO vo) {
        final DataTemplate template = dataImportService.getById(vo.getId());
        final List<DataField> fieldList = dataFieldService.list(Wrappers.lambdaQuery(DataField.class)
                .eq(DataField::getTemplateId, template.getId())
                .isNotNull(DataField::getField)
                .select(DataField::getField));
        final String fieldSql = fieldList.stream().map(DataField::getField).collect(Collectors.joining(",")) + ",YEAR";
        try {
            DataSourceUtil.switchDs(template.getDataSourceId());
            // 先查询出来所有的数据
            String selectSql = "SELECT " + fieldSql + " FROM " + template.getTableName() + "_TEMP";
            log.info("查询sql为:[{}]", selectSql);
            final List<Map<String, Object>> dataList = jdbcTemplate.queryForList(selectSql);
            if (CollectionUtils.isEmpty(dataList)) {
                return AjaxResult.error("临时表数据为空");
            }
            String[] sqlArr = new String[dataList.size()];
            for (int i = 0; i < dataList.size(); i++) {
                StringBuilder sb = new StringBuilder();
                sb.append("INSERT INTO ").append(template.getTableName()).append("(").append(fieldSql).append(") VALUES(");
                final Map<String, Object> objectMap = dataList.get(i);
                for (DataField field : fieldList) {
                    final Object o = objectMap.get(field.getField());
                    if (o == null) {
                        sb.append("'',");
                    } else {
                        sb.append("'").append(o).append("',");
                    }
                }
                sb.append("'").append(template.getYear()).append("'");
                sb.append(")");
                log.info("插入数据sql为:[{}]", sb);
                sqlArr[i] = sb.toString();
            }
            jdbcTemplate.batchUpdate(sqlArr);
            // 导入完毕直接清空临时表
            jdbcTemplate.execute("DELETE FROM " + template.getTableName() + "_TEMP");
        } catch (Exception e) {
            throw new BaseException(e.getMessage());
        } finally {
            DataSourceUtil.switchDefaultDs();
        }
        DataTemplate updateTemplate = new DataTemplate();
        updateTemplate.setId(template.getId());
        updateTemplate.setImportStatus("1");
        updateTemplate.setImportTime(new Date());
        dataImportService.updateById(updateTemplate);
        return AjaxResult.success();
    }


}
