package com.tbyf.his.job.task;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.tbyf.his.apiconvert.domain.ApiconvertBaseInfoModel;
import com.tbyf.his.apiconvert.service.IApiconvertBaseinfoService;
import com.tbyf.his.common.core.domain.AjaxResult;
import com.tbyf.his.framework.datasource.DataSourceUtil;
import com.tbyf.his.framework.system.service.ISysConfigService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.task.TaskExecutor;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import org.springframework.web.client.RestTemplate;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;

/**
 * 就诊记录调度任务
 *
 * @author lzz
 * @date 2022/11/24 14:50
 */
@Component("medicalRecordTask")
@Slf4j
public class MedicalRecordTask {

    @Autowired
    private IApiconvertBaseinfoService apiconvertBaseinfoService;

    @Autowired
    private ISysConfigService iSysConfigService;

    @Autowired
    private RestTemplate restTemplate;

    @Autowired
    private JdbcTemplate jdbcTemplate;

    @Autowired
    private TaskExecutor taskExecutor;

    public static AtomicLong DATA_INDEX = new AtomicLong(0);

    private static Boolean DATA_EXECUTE = true;

    private static final String[] FIELDS = new String[]{
            "patientId", "clinicClass", "patientName", "sex", "age", "patientProperties",
            "brithDate", "orgCode", "deptSN", "deptName", "clinicDate", "jzlsh", "completionDate",
            "primaryDiagnosis", "auxiliaryDiagnosis", "orgName", "rowCode"
    };

    private static final String SQL = "INSERT INTO T_MEDICAL_RECORD(patientId,clinicClass,patientName,sex,age,patientProperties,brithDate,orgCode,deptSN,deptName,clinicDate,jzlsh,completionDate,primaryDiagnosis,auxiliaryDiagnosis,orgName,rowCode,empiId)VALUES(?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?)";

    /**
     * 存储运行情况,key为orgCode+type类型 value为数量
     */
    public static final Map<String, AtomicLong> INDEX_MAP = new HashMap<>();

    public static final Map<String, AtomicBoolean> EXECUTE_MAP = new HashMap<>();

    /**
     * 重置调度任务
     *
     * @param orgCode
     * @param num1         1门诊 2住院
     * @param num2         如果是-1,就查询当前数量重置  如果是-2,就只展示信息
     * @param dataCenterId
     */
    public void reset(String orgCode, String num1, String num2, String dataCenterId) {
        String key = orgCode + "-" + num1;
        if (!"-2".equals(num2)) {
            long count = Long.parseLong(num2);
            if ("-1".equals(num2)) {
                count = getCount(orgCode, num1, dataCenterId);
            }
            AtomicLong atomicLong = INDEX_MAP.get(key);
            if (atomicLong == null) {
                atomicLong = new AtomicLong(count);
                INDEX_MAP.put(key, atomicLong);
            } else {
                atomicLong.set(count);
            }
        }
        log.info("======================就诊记录调度重置=======================");
        log.info("======================当前各任务就诊记录执行情况为:[{}]=======================", JSON.toJSONString(INDEX_MAP));
        log.info("======================当前各任务就诊记录调度情况为:[{}]=======================", JSON.toJSONString(EXECUTE_MAP));
    }

    public static final String COUNT_SQL = "SELECT COUNT(jzlsh) FROM T_MEDICAL_RECORD WHERE orgCode = ? and clinicClass = ?";

    /**
     * 记录条数
     *
     * @param orgCode
     * @param clinicClass  1-门诊 2-住院
     * @param dataCenterId
     * @return
     */
    public long getCount(String orgCode, String clinicClass, String dataCenterId) {
        try {
            DataSourceUtil.switchDs(dataCenterId);
            final Long aLong = jdbcTemplate.queryForObject(COUNT_SQL, new Object[]{orgCode, clinicClass}, Long.class);
            if (aLong == null) {
                return 0;
            }
            return aLong;
        } catch (Exception e) {
            return 0;
        } finally {
            DataSourceUtil.switchDefaultDs();
        }
    }

    public long getIndex(String orgCode, String clinicClass, String dataCenterId) {
        String key = orgCode + "-" + clinicClass;
        AtomicLong atomicLong = INDEX_MAP.get(key);
        if (atomicLong == null) {
            atomicLong = new AtomicLong(getCount(orgCode, clinicClass, dataCenterId));
            INDEX_MAP.put(key, atomicLong);
        }
        return atomicLong.get();
    }

    public void addIndex(String orgCode, String clinicClass, long num) {
        String key = orgCode + "-" + clinicClass;
        AtomicLong atomicLong = INDEX_MAP.get(key);
        if (atomicLong == null) {
            atomicLong = new AtomicLong(num);
            INDEX_MAP.put(key, atomicLong);
        }
        atomicLong.addAndGet(num);
    }

    /**
     * 当前是否可执行
     *
     * @param orgCode
     * @return
     */
    public boolean isExecute(String orgCode) {
        AtomicBoolean aBoolean = EXECUTE_MAP.get(orgCode);
        if (aBoolean != null) {
            return aBoolean.get();
        } else {
            aBoolean = new AtomicBoolean(true);
            EXECUTE_MAP.put(orgCode, aBoolean);
            return aBoolean.get();
        }
    }

    /**
     * 修改执行状态
     *
     * @param orgCode
     * @param init
     */
    public void resetExecute(String orgCode, boolean init) {
        AtomicBoolean aBoolean = EXECUTE_MAP.get(orgCode);
        if (aBoolean != null) {
            aBoolean.set(init);
        } else {
            aBoolean = new AtomicBoolean(init);
            EXECUTE_MAP.put(orgCode, aBoolean);
        }
    }

    /**
     * 就诊记录调度
     * CREATE TABLE T_MEDICAL_RECORD
     * (
     * patientId          VARCHAR2(32),
     * clinicClass        VARCHAR2(32),
     * patientName        VARCHAR2(50),
     * sex                VARCHAR2(10),
     * age                NUMBER,
     * patientProperties  VARCHAR2(80),
     * brithDate          VARCHAR2(20),
     * orgCode            VARCHAR2(10),
     * deptSN             NUMBER,
     * deptName           VARCHAR2(50),
     * clinicDate         VARCHAR2(19),
     * jzlsh              VARCHAR2(32),
     * completionDate     VARCHAR2(19),
     * primaryDiagnosis   VARCHAR2(4000),
     * auxiliaryDiagnosis VARCHAR2(4000),
     * orgName            VARCHAR2(60),
     * empiId             VARCHAR2(60),
     * rowCode            VARCHAR2(60)
     * )
     */
    public void dataExtractionTask(String apiId1, String apiId2, String orgCode, String size, String dataSourceId) {
        if (isExecute(orgCode)) {
            resetExecute(orgCode, false);
            final long startTime = System.currentTimeMillis();
            final String empiUrl = iSysConfigService.selectConfigByKey("EMPI_URL") + "/open/api/handlerRequestNoAuth/empiId";
            final String empiUrl2 = iSysConfigService.selectConfigByKey("EMPI_URL") + "/open/api/handlerRequestNoAuth/empiIdByIdCardName";
            final CompletableFuture<Void> future1 = CompletableFuture.runAsync(() -> {
                try {
                    long index = getIndex(orgCode, "1", dataSourceId);
                    final long start = index + 1;
                    final long end = index + Long.parseLong(size);
                    log.info("====================门诊就诊记录[{}]调度查询索引:[{}],[{}]====================", orgCode, start, end);
                    final ApiconvertBaseInfoModel baseInfoModel = new ApiconvertBaseInfoModel();
                    baseInfoModel.setApiId(Long.parseLong(apiId1));
                    final JSONObject param = new JSONObject();
                    param.put("start", start);
                    param.put("end", end);
                    baseInfoModel.setParam(param.toJSONString());
                    baseInfoModel.setOrgCode(Long.parseLong(orgCode));
                    DataSourceUtil.switchDefaultDs();
                    List<JSONObject> jsonArray = (List<JSONObject>) apiconvertBaseinfoService.handlerRequest(baseInfoModel);
                    log.info("====================门诊就诊记录[{}]调度,实际查询调度数量为:[{}]====================", orgCode, jsonArray.size());
                    if (CollectionUtils.isEmpty(jsonArray)) {
                        log.info("====================门诊就诊记录[{}]已无新数据,调度完毕,请适当关闭任务====================", orgCode);
                    } else {
                        List<Object[]> batchList = new ArrayList<>();
                        final List<JSONObject> requestList = jsonArray.stream().map(item -> {
                            final JSONObject object = new JSONObject();
                            object.put("orgCode", item.getString("orgCode"));
                            object.put("sysCode", "99");
                            object.put("sysRECId", item.getString("id"));
                            return object;
                        }).collect(Collectors.toList());
                        final AjaxResult ajaxResult = restTemplate.postForEntity(empiUrl, requestList, AjaxResult.class).getBody();
                        List<String> empiList = (List<String>) ajaxResult.get(AjaxResult.DATA_TAG);
                        for (int i = 0; i < jsonArray.size(); i++) {
                            final JSONObject jsonObject = jsonArray.get(i);
                            Object[] objects = new Object[FIELDS.length + 1];
                            for (int j = 0; j < FIELDS.length; j++) {
                                objects[j] = jsonObject.get(FIELDS[j]);
                            }
                            objects[FIELDS.length] = empiList.get(i);
                            batchList.add(objects);
                        }
                        DataSourceUtil.switchDs(dataSourceId);
                        jdbcTemplate.batchUpdate(SQL, batchList);
                        addIndex(orgCode, "1", batchList.size());
                    }
                } catch (Exception e) {
                    log.error("就诊记录调度执行失败", e);
                } finally {
                    DataSourceUtil.switchDefaultDs();
                }
            }, taskExecutor);

            final CompletableFuture<Void> future2 = CompletableFuture.runAsync(() -> {
                try {
                    long index = getIndex(orgCode, "2", dataSourceId);
                    final long start = index + 1;
                    final long end = index + Long.parseLong(size);
                    log.info("====================住院就诊记录[{}]调度查询索引:[{}],[{}]====================", orgCode, start, end);
                    final ApiconvertBaseInfoModel baseInfoModel = new ApiconvertBaseInfoModel();
                    baseInfoModel.setApiId(Long.parseLong(apiId2));
                    final JSONObject param = new JSONObject();
                    param.put("start", start);
                    param.put("end", end);
                    baseInfoModel.setParam(param.toJSONString());
                    baseInfoModel.setOrgCode(Long.parseLong(orgCode));
                    DataSourceUtil.switchDefaultDs();
                    List<JSONObject> jsonArray = (List<JSONObject>) apiconvertBaseinfoService.handlerRequest(baseInfoModel);
                    log.info("====================住院就诊记录[{}]调度,实际查询调度数量为:[{}]====================", orgCode, jsonArray.size());
                    if (CollectionUtils.isEmpty(jsonArray)) {
                        log.info("====================住院就诊记录[{}]已无新数据,调度完毕,请适当关闭任务====================", orgCode);
                    } else {
                        List<Object[]> batchList = new ArrayList<>();
                        final List<JSONObject> requestList = jsonArray.stream().map(item -> {
                            final JSONObject object = new JSONObject();
                            object.put("name", item.getString("patientName"));
                            object.put("idNo", item.getString("id"));
                            return object;
                        }).collect(Collectors.toList());
                        final AjaxResult ajaxResult = restTemplate.postForEntity(empiUrl2, requestList, AjaxResult.class).getBody();
                        List<String> empiList = (List<String>) ajaxResult.get(AjaxResult.DATA_TAG);
                        for (int i = 0; i < jsonArray.size(); i++) {
                            final JSONObject jsonObject = jsonArray.get(i);
                            Object[] objects = new Object[FIELDS.length + 1];
                            for (int j = 0; j < FIELDS.length; j++) {
                                objects[j] = jsonObject.get(FIELDS[j]);
                            }
                            objects[FIELDS.length] = empiList.get(i);
                            batchList.add(objects);
                        }
                        DataSourceUtil.switchDs(dataSourceId);
                        jdbcTemplate.batchUpdate(SQL, batchList);
                        addIndex(orgCode, "2", batchList.size());
                    }
                } catch (Exception e) {
                    log.error("就诊记录调度执行失败", e);
                } finally {
                    DataSourceUtil.switchDefaultDs();
                }
            }, taskExecutor);
            CompletableFuture.allOf(future1, future2).whenComplete((s, ex) -> {
                log.info("========================就诊记录调度任务,orgCode:[{}]:执行时间:[{}],执行情况:[{}]=======================", orgCode, System.currentTimeMillis() - startTime, JSON.toJSONString(INDEX_MAP));
                resetExecute(orgCode, true);
            });
        } else {
            log.info("====================上次就诊记录调度未执行完毕,当前执行数据为:orgCode:[{}],[{}}]====================", orgCode, JSON.toJSONString(INDEX_MAP));
        }
    }


}
