package com.tbyf.his.web.controller.dataImport;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.tbyf.his.common.annotation.IgnoreWebSecurity;
import com.tbyf.his.common.core.domain.AjaxResult;
import com.tbyf.his.common.core.page.TableDataInfo;
import com.tbyf.his.common.utils.StringUtils;
import com.tbyf.his.common.utils.bean.BeanUtils;
import com.tbyf.his.web.dataImport.core.DiConfig;
import com.tbyf.his.web.dataImport.core.RuleVO;
import com.tbyf.his.web.dataImport.core.RuleValidator;
import com.tbyf.his.web.dataImport.domain.param.AddRuleParam;
import com.tbyf.his.web.dataImport.domain.param.BindRuleParam;
import com.tbyf.his.web.dataImport.domain.param.QueryRuleParam;
import com.tbyf.his.web.dataImport.domain.param.UpdateRuleParam;
import com.tbyf.his.web.dataImport.entity.BindRule;
import com.tbyf.his.web.dataImport.entity.DataField;
import com.tbyf.his.web.dataImport.entity.DataRule;
import com.tbyf.his.web.dataImport.entity.DataTemplate;
import com.tbyf.his.web.dataImport.service.BindRuleService;
import com.tbyf.his.web.dataImport.service.DataFieldService;
import com.tbyf.his.web.dataImport.service.DataRuleService;
import com.tbyf.his.web.dataImport.service.DataTemplateService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import java.util.*;
import java.util.stream.Collectors;

/**
 * @author lzz
 * @date 2023/2/16 15:29
 */
@RestController
@Api(tags = "数据字段接口")
@RequestMapping("/data/rule")
@Slf4j
public class DataRuleController {

    @Autowired
    private DataRuleService dataRuleService;

    @Autowired
    private BindRuleService bindRuleService;

    @Autowired
    private DataFieldService dataFieldService;

    @Autowired
    private DataTemplateService dataTemplateService;

    @IgnoreWebSecurity
    @GetMapping("")
    @ApiOperation("规则查询")
    public TableDataInfo queryRule(@Validated QueryRuleParam param) {
        final Page<DataRule> page = Page.of(param.getPageNum(), param.getPageSize());
        final LambdaQueryWrapper<DataRule> queryWrapper = Wrappers.lambdaQuery(DataRule.class);
        queryWrapper.like(StringUtils.isNotBlank(param.getName()), DataRule::getName, param.getName())
                .eq(StringUtils.isNotBlank(param.getType()), DataRule::getType, param.getType())
                .in(DataRule::getType, "基础规则", "组合规则");
        final Page<DataRule> templatePage = dataRuleService.page(page, queryWrapper);
        return param.convert(templatePage);
    }

    @IgnoreWebSecurity
    @GetMapping("/dict")
    @ApiOperation("规则字典")
    public AjaxResult queryRuleDict() {
        final LambdaQueryWrapper<DataRule> queryWrapper = Wrappers.lambdaQuery(DataRule.class);
        queryWrapper.in(DataRule::getType, "基础规则", "组合规则")
                .select(DataRule::getId, DataRule::getName);
        return AjaxResult.success(dataRuleService.list(queryWrapper));
    }

    @IgnoreWebSecurity
    @PostMapping("")
    @ApiOperation("规则新增")
    public AjaxResult addRule(@RequestBody @Validated AddRuleParam param) {
        DataRule rule = new DataRule();
        BeanUtils.copyProperties(param, rule);
        dataRuleService.save(rule);
        return AjaxResult.success();
    }

    @IgnoreWebSecurity
    @PostMapping("/update")
    @ApiOperation("修改规则")
    public AjaxResult updateRule(@RequestBody @Validated UpdateRuleParam param) {
        DataRule rule = new DataRule();
        BeanUtils.copyProperties(param, rule);
        dataRuleService.updateById(rule);
        return AjaxResult.success();
    }

    @IgnoreWebSecurity
    @GetMapping("/delete")
    @ApiOperation("删除规则")
    public AjaxResult deleteRule(@RequestParam String ruleId) {
        dataRuleService.removeById(ruleId);
        bindRuleService.remove(Wrappers.lambdaQuery(BindRule.class).eq(BindRule::getRuleId, ruleId));
        return AjaxResult.success();
    }

    @IgnoreWebSecurity
    @PostMapping("/test")
    @ApiOperation("规则测试")
    public AjaxResult testRule(@RequestBody @Validated RuleVO test) {
        final RuleValidator validator = DiConfig.getValidator(test.getMode());
        validator.validate(test);
        if (StringUtils.isNotBlank(test.getResult())) {
            return AjaxResult.error(test.getResult());
        }
        return AjaxResult.success();
    }

    @IgnoreWebSecurity
    @GetMapping("/bind/quick")
    @ApiOperation("快速绑定")
    public AjaxResult quickBind(@RequestParam String templateId,
                                @RequestParam String ruleId) {
        final LambdaQueryWrapper<DataField> queryWrapper = Wrappers.lambdaQuery(DataField.class)
                .select(DataField::getId)
                .eq(DataField::getTemplateId, templateId);
        final List<String> fieldIdList = dataFieldService.list(queryWrapper).stream()
                .map(DataField::getId)
                .collect(Collectors.toList());
        final LambdaQueryWrapper<BindRule> removeWrapper = Wrappers.lambdaQuery(BindRule.class)
                .in(BindRule::getDataId, fieldIdList)
                .eq(BindRule::getRuleId, ruleId);
        bindRuleService.remove(removeWrapper);
        final List<BindRule> bindRuleList = fieldIdList.stream().map(item -> {
            BindRule bind = new BindRule();
            bind.setDataId(item);
            bind.setRuleId(ruleId);
            return bind;
        }).collect(Collectors.toList());
        bindRuleService.saveBatch(bindRuleList);
        return AjaxResult.success();
    }

    @IgnoreWebSecurity
    @GetMapping("/bind")
    @ApiOperation("查询绑定的规则")
    public AjaxResult queryBindRule(@RequestParam String fieldId) {
        final List<BindRule> list = bindRuleService.list(Wrappers.lambdaQuery(BindRule.class)
                .eq(BindRule::getDataId, fieldId));
        if (CollectionUtils.isEmpty(list)) {
            return AjaxResult.success(Collections.emptyList());
        }
        final List<String> idList = list.stream().map(BindRule::getRuleId).collect(Collectors.toList());
        return AjaxResult.success(dataRuleService.list(Wrappers.lambdaQuery(DataRule.class)
                .in(DataRule::getId, idList)));
    }

    @IgnoreWebSecurity
    @PostMapping("/bind")
    @ApiOperation("配置字段规则绑定")
    @Transactional(rollbackFor = Exception.class)
    public AjaxResult bindRule(@RequestBody BindRuleParam param) {
        bindRuleService.remove(Wrappers.lambdaQuery(BindRule.class).eq(BindRule::getDataId, param.getFieldId()));
        if (!CollectionUtils.isEmpty(param.getRuleIdList())) {
            final List<BindRule> bindRuleList = param.getRuleIdList().stream().map(item -> {
                BindRule bd = new BindRule();
                bd.setRuleId(item);
                bd.setDataId(param.getFieldId());
                return bd;
            }).collect(Collectors.toList());
            bindRuleService.saveBatch(bindRuleList);
        }
        return AjaxResult.success();
    }

    @IgnoreWebSecurity
    @GetMapping("/sync/year")
    @ApiOperation("根据年份同步规则")
    public AjaxResult syncRuleByYear(@RequestParam String templateId, @RequestParam String year) {
        // 查询两个模板
        DataTemplate sourceTemplate = dataTemplateService.getById(templateId);
        DataTemplate targetTemplate = dataTemplateService.getOne(Wrappers.lambdaQuery(DataTemplate.class).eq(DataTemplate::getYear, year).eq(DataTemplate::getOrgName, sourceTemplate.getOrgName()), false);
        List<DataField> fieldList = dataFieldService.list(Wrappers.lambdaQuery(DataField.class)
                .eq(DataField::getTemplateId, sourceTemplate.getId())
                .isNotNull(DataField::getField)
                .select(DataField::getField, DataField::getId));
        List<DataField> targetFieldList = dataFieldService.list(Wrappers.lambdaQuery(DataField.class)
                .eq(DataField::getTemplateId, targetTemplate.getId())
                .isNotNull(DataField::getField)
                .select(DataField::getField, DataField::getId));
        if (!CollectionUtils.isEmpty(fieldList)) {
            List<BindRule> rules = new ArrayList<>();
            List<BindRule> sourceBindList = bindRuleService.queryRuleList(sourceTemplate.getId());
            List<BindRule> targetBindList = bindRuleService.queryRuleList(targetTemplate.getId());
            // 字段规则同步
            if (!CollectionUtils.isEmpty(targetBindList)) {
                for (BindRule targetRule : targetBindList) {
                    Optional<DataField> first = fieldList.stream().filter(df -> StringUtils.equals(df.getField(), targetRule.getDataId())).findFirst();
                    if (first.isPresent()) {
                        boolean match = sourceBindList.stream()
                                .anyMatch(br -> StringUtils.equals(br.getDataId(), targetRule.getDataId()) && StringUtils.equals(br.getRuleId(), targetRule.getRuleId()));
                        if (!match) {
                            rules.add(new BindRule(first.get().getId(), targetRule.getRuleId()));
                        }
                    }
                }
            }
            // 模板规则同步
            List<DataRule> sourceRules = dataRuleService.queryTemplateRules(sourceTemplate.getId());
            List<DataRule> targetRules = dataRuleService.queryTemplateRules(targetTemplate.getId());
            if (!CollectionUtils.isEmpty(targetRules)) {
                List<DataRule> datas = new ArrayList<>();
                root:
                for (DataRule targetRule : targetRules) {
                    String[] split = targetRule.getContent().split("=");
                    String[] split1 = split[0].split(",");
                    String[] split2 = split[1].split(",");
                    for (int i = 0; i < split1.length; i++) {
                        int finalI = i;
                        Optional<DataField> first = targetFieldList.stream().filter(item -> StringUtils.equals(item.getId(), split1[finalI])).findFirst();
                        if (first.isPresent()) {
                            Optional<DataField> first1 = fieldList.stream().filter(item -> StringUtils.equals(item.getField(), first.get().getField())).findFirst();
                            if (first1.isPresent()) {
                                split1[i] = first1.get().getId();
                            } else {
                                continue root;
                            }
                        } else {
                            continue root;
                        }
                    }
                    for (int j = 0; j < split2.length; j++) {
                        int finalI = j;
                        Optional<DataField> first = targetFieldList.stream().filter(item -> StringUtils.equals(item.getId(), split2[finalI])).findFirst();
                        if (first.isPresent()) {
                            Optional<DataField> first1 = fieldList.stream().filter(item -> StringUtils.equals(item.getField(), first.get().getField())).findFirst();
                            if (first1.isPresent()) {
                                split2[j] = first1.get().getId();
                            } else {
                                continue root;
                            }
                        } else {
                            continue root;
                        }
                    }
                    targetRule.setId(null);
                    targetRule.setContent(String.join(",", split1) + "=" + String.join(",", split2));
                    datas.add(targetRule);
                }
                if (!CollectionUtils.isEmpty(datas)) {
                    dataRuleService.saveBatch(datas);
                    datas.forEach(item -> {
                        BindRule rule = new BindRule();
                        rule.setDataId(sourceTemplate.getId())
                                .setRuleId(item.getId());
                        rules.add(rule);
                    });
                }
            }
            if (!CollectionUtils.isEmpty(rules)) {
                bindRuleService.saveBatch(rules);
            }
        }
        return AjaxResult.success();
    }


}
